(**   *)
(*
  Panoptes: An Exploration Tool for Formal Proofs
  Copyright (C) 2008  Orlin Grigorov

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

  To contact the author, email to: ogrigorov at gmail.com
*)


open Node

(** This class is instantiated to represent an edge in the deduction graph in Panoptes.  Complying with the properties of the graph, each edge is associated with two different nodes.  *)
class arrow (from_Node, to_Node, control_points)=
object (self)

(** A reference pointer to the object representing the originating node of the arrow.  *)
  val origin:node = from_Node

(** A reference pointer to the object representing the destination node of the arrow (the node that the arrow points at).  *)
  val destination:node = to_Node

(** This is a list of control points for the edge.  This method consequently became obsolete: these points are used for drawing bezier curves, which were disabled at the point in the implementation when the functionality for allowing manual repositioning of the nodes in the graph was introduced.  Before that, these control points were being generated upon deduction graph initialization by the program that generates the automatic layout for the graph (Graphviz).  *)
  val mutable pairs = control_points

  method get_origin = origin
(** These two methods return the reference pointers to the objects that represent the origin and destination nodes respectively.  *)
  method get_destination = destination

(** This method draws (displays) the arrow in the OpenGL drawing space.  *)
  method draw () =
    GlMat.push();

    if (destination#get_path_flag && origin#get_path_flag) then 
      begin
	GlDraw.color ~alpha:1.0 (1., 0., 0.);
	GlDraw.line_width (1.8);
      end
    else 
      begin
	GlDraw.color ~alpha:1.0 (0., 0., 1.);
	GlDraw.line_width (1.2);
      end;

    
    let lst:(float list ref) = ref [] in
      
    pairs <- [];  (* Bezier override *)

    if (List.length pairs > 0) then
      begin
	(* ========= Draw the Bezier curved edges =======================*)
	GlDraw.shade_model `flat;  (* could be `smooth or `flat *)
	Gl.enable `map1_vertex_3;

	let dummy = function
	    (xx, yy) -> lst := !lst @ [origin#calc_x xx] @ [origin#calc_y yy] @ [origin#get_z -. 0.1]
	in 
	  List.iter dummy pairs;
     
	let tmp = Raw.of_float_array (Array.of_list !lst) ~kind:`double in
	GlMap.map1 ~target:(`vertex_3) (0.0, 1.0) ~order:(List.length pairs) tmp;
	    

	GlDraw.begins `line_strip;
	(*    GlDraw.vertex3 (origin#get_x, origin#get_y, origin#get_z -. 0.1); *)
	for i = 0 to 100 do
	  GlMap.eval_coord1 (float_of_int(i) /. 100.0);
	done;
	GlDraw.vertex3 (destination#get_x, destination#get_y, destination#get_z -. 0.1);
	GlDraw.ends ();
	    
	Gl.disable `map1_vertex_3;
	(* ========== END drawing Bezier curved edges =================== *)
      end
    else
      begin
	let (tmp_x, tmp_y, tmp_z) as tmp_vertex = (destination#get_x, 
						   destination#get_y, 
						   destination#get_z -. 0.1) in
	lst := !lst @ [tmp_x; tmp_y; tmp_z];
	GlDraw.begins `lines;
	GlDraw.vertex3 (origin#get_x, origin#get_y, origin#get_z -. 0.1);
	GlDraw.vertex3 tmp_vertex;
	GlDraw.ends();
      end;


    (* ========== Drawing ARROW HEAD ================================ *)

    let x2, y2 = List.hd !lst, List.hd (List.tl !lst) in
    let alpha =
      let x1 = origin#get_x and y1 = origin#get_y in
      let tmp_l = sqrt((x2 -. x1) *. (x2 -. x1) +. (y2 -. y1) *. (y2 -. y1))
      and tmp_x = (x1 -. x2) in 
      let beta = (acos(tmp_x /. tmp_l) *. 180. /. 3.14156) in
	(match (y1 <= y2) with
	     true -> -1. *. beta
	   | false -> beta
	)
    in

    GlMat.pop();
    GlMat.push();

    (*========== INTERSECTION TESTING =============== *)
    let is_intersecting (x1, y1, x2, y2) (x3, y3, x4, y4) =
      let denominator = ((y4 -. y3) *. (x2 -. x1) -. (x4 -. x3) *. (y2 -. y1)) in
	if (denominator = 0.) then (false, 0., 0.)  (* segments are parallel *)
	else
	  begin
	    let u_a =	((x4 -. x3) *. (y1 -. y3) -. (y4 -. y3) *. (x1 -. x3)) /. denominator
	    and u_b =	((x2 -. x1) *. (y1 -. y3) -. (y2 -. y1) *. (x1 -. x3)) /. denominator
	    in
	    let x = x1 +. u_a *. (x2 -. x1)
	    and y = y1 +. u_a *. (y2 -. y1)
	    in
	      if (u_a > 0.) && (u_a < 1.) && (u_b > 0.) && (u_b < 1.) then (true, x, y)
	      else (false, 0., 0.)
	  end
    in  


    let x2 = ref x2 
    and y2 = ref y2 
    and o_h = origin#get_height /. 2.
    and o_w = origin#get_width /. 2.
    and o_x = origin#get_x
    and o_y = origin#get_y
    and d_x = destination#get_x
    and d_y = destination#get_y
    in
    let test_intersection (a, b, c, d) =
      let (result, intersect_x, intersect_y) =
	is_intersecting (o_x, o_y, d_x, d_y) (a, b, c, d)
      in
	if (result) then (x2 := intersect_x; y2 := intersect_y);
	(result)
    in

    if (false = test_intersection (o_x -. o_w, o_y -. o_h, o_x +. o_w, o_y -. o_h)) then
      if (false = test_intersection (o_x -. o_w, o_y +. o_h, o_x +. o_w, o_y +. o_h)) then
	if (false = test_intersection (o_x -. o_w, o_y +. o_h, o_x -. o_w, o_y -. o_h)) then
	  if (false = test_intersection (o_x +. o_w, o_y +. o_h, o_x +. o_w, o_y -. o_h)) then ();


    (* ============= end INTERSECTION TESTING ============= *)
      
    GlMat.translate3 (!x2, !y2, origin#get_z +. 0.1);
    GlMat.scale ~x:(origin#get_height) ~y:(origin#get_height) (); 

    GlMat.rotate3 ~angle:(alpha) (0.0, 0.0, 1.0);

    GlDraw.begins `triangles;

(*
    GlDraw.vertex3 (0.4, 0., 0.);
    GlDraw.vertex3 (-0.6, 0.2, 0.);
    GlDraw.vertex3 (-0.6, -0.2, 0.);
*)
    GlDraw.vertex3 (0., 0., 0.);
    GlDraw.vertex3 (-1., 0.2, 0.);
    GlDraw.vertex3 (-1., -0.2, 0.);


    GlDraw.ends();

(* ========== END Drawing ARROW HEAD ============================ *)



    GlMat.pop();
    ()


end;;
