(**   *)
(*
  Panoptes: An Exploration Tool for Formal Proofs
  Copyright (C) 2008  Orlin Grigorov

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

  To contact the author, email to: ogrigorov at gmail.com
*)


open Node
open Arrow
open Parser_dot
open Types
open TextGL

(**/**)
exception Error of string;;
(**/**)

(** This whole class can be considered as the main Panoptes engine.  Most of the user induced operations are implemented in this class.  In addition, the class is responsible for coordinating and invoking all drawing procedures used for visualizing all possible components of the deduction graph (nodes, arrows, paths, information boxes, collapsed boxes, help screen, new data notification, etc.).  *)
class view dg = 
object (self)

  val mutable screenWidth = 800.
(** These two variables comprise the current resolution of the OpenGL window.  Upon initial initialization, the size is 800 pixels by 600 pixels by default (although the user is allowed to resize the window without limitations).  *)
  val mutable screenHeight = 600.

(** This variable holds the field of view (FOV) angle in degrees.  Refer to the OpenGL documentation of the Implementation chapter of this thesis to find more explanation about FOV and how it works.  *)
  val fovy = 65.

  val mutable x1 = 0.0
  val mutable y1 = 0.0
(** These variables specify the coordinates of the absolute center of the space (this space can be viewed as the complete universe, in which the deduction graph resides).  The location of every single component from a mere point to a large and complicated shape has certain coordinates that are relative to the coordinates represented by these three variables.  Of course, since we do not utilize the ability to move this center of space to achieve certain effects in Panoptes, the coordinates are conveniently reset to zeroes.   *)
  val mutable z1 = 0.0

(** The initial [z] coordinate of the deduction graph.  It must be deep in space to be visible (away from the user) in order to be visible and within sight (the sight is defined by the FOV and other parameters, which are described later).  *)
  val init_z = -520.0

(** A list of tuples.  Each tuple contains a node name (in string format) and a reference to a created object of the [Node] class, which is associated with that node.  This list comprise all nodes of the deduction graph, which were created during initialization and are available for the program for direct OpenGL display.   *)
  val mutable nodes = []

(** A list of all created arrows (edges). Similarly to the previous list, this list contains all arrows in the deduction graph, which were created during initialization and are available for direct OpenGL display.  *)
  val mutable arrows = []

(** Instantiation of [Parser]. In fact notice that this object was passed to the current class upon initialization of this class.  Therefore, this variable keeps a reference pointer to object that was already created.  *)
  val mutable parser_dg = dg

(** A new instantiation of [Parser_dot] that will be used for running the deduction graph through the layout generation program GraphViz.  Notice how the current class serves as a bridge between all other classes.  In this case, the previous [parser_dg] will collect all information that describes the deduction graph.  The current class will then retrieve this information and send it to [dedotter] for processing, which in turn will return the suggested by Graphviz locations of all nodes.  *)
  val mutable dedotter = new parser_dot

(** This is needed by OpenGL in order to implement the selection technique.  Usually, each object in the window is assigned a ``selection value'' upon instantiation.  A special function then prompts OpenGL to run an imaginary linear ray from a certain point in space towards a certain direction.  When this ray hits a target, the selection value of that target is returned.  This value can then be matched against the tuples of this hash table to discover the name of the object that was hit.  Of course, this object can be any possible element of the deduction graph that has a selection name assigned to it.  In our case, such objects are the nodes and the information boxes, although the arrows are assigned a unique, but the same for all arrows, for facilitating the process of debugging.  This selection values technique is hereby used to detect the object the user clicks on.  *)
  val mutable selection_names = Hashtbl.create 0

(** When the user presses down and holds the left mouse button over a certain node (notice that this class provides the functionality, while the actual event listeners are implemented in the [Panoptes] class), then the boolean of this variable becomes TRUE until the user releases the button.  The other three values keep the 3D coordinates of the node at the moment the mouse button was pressed. Other methods, described below, access the values of this tuple every time a new screen frame must be drawn.  If the first element is true, these other methods perform appropriate actions.  *)
  val mutable locked_node = (false, "", 0., 0., 0.)

(** Similar to the above variable, this variable signifies a lock on the whole deduction graph.  Its purpose is mainly used to notify the other methods that the user is dragging the graph with the mouse.  *)
  val mutable locked_dg_drag = (false, 0., 0.)

(** This variable keeps the [z] coordinates of all open (visible) content boxes.  As such, some content boxes will be closer to the screen while others will be further away.  This is important to achieve control over the focus of the content box that the user wants to be completely visible, especially when the content boxes are arranged in an overlapping manner.  *)    
  val mutable content_box_ordering = Hashtbl.create 0

(** This variable keeps a list of all newly added arrows.  Such arrows are usually added when a collapsing procedure is performed.  Recall the Design chapter, which stated that all collapsing procedures produce new nodes. It is logical that if there is a new node, it will be connected to other nodes through new links (in this thesis we usually call these links arrows).  Furthermore, to facilitate the reversibility of all user induced actions (as requested by the Requirements chapter), these new arrows are conveniently stored in this separate list.  Thus the original arrangement of the graph is always preserved.  *)
  val mutable added_arrows = []

(** A hash table that keeps information about all created by that point collapsed nodes.  Each element of the hash table contains information about the nodes it hides, the arrows it is associated with, as well as other useful information.  *)
  val mutable collapsed = Hashtbl.create 0

(** This method resets all variables of this class.  It is called when the user decides to reset Panoptes, which usually happens when the development of a new proof is started in IMPS.  *)
  method reset_all_data =
    x1 <- 0.0;
    y1 <- 0.0;
    z1 <- 0.0;
    nodes <- [];
    arrows <- [];
    dedotter <- new parser_dot;
    selection_names <- Hashtbl.create 0;
    Hashtbl.add selection_names "arrow" 1;    
    locked_node <- (false, "", 0., 0., 0.);
    locked_dg_drag <- (false, 0., 0.);
    content_box_ordering <- Hashtbl.create 0;
    added_arrows <- [];
    collapsed <- Hashtbl.create 0;
    ()

  initializer
    self#init();
    ()

  (*=========================================================*)
(** This method resets the location and zoom level of the deduction graph to the screen according to the generated initial layout.  This is useful when the user manually relocates the deduction graph and later decides to revert back and fit everything in the screen.  *)
  method fit_to_screen ()=
    x1 <- 0.0;
    y1 <- 0.0;
    z1 <- 0.0;
    self#draw();
    ()

(** This method builds the deduction graph.  It creates instances of [Node] and [Arrow] for each graph node and edge.  Verbose information is displayed in the command shell during this process, as this may sometimes take more than a second for graphs that contain hundreds of nodes.  Additional things that are accomplished by this method include calculation and initialization of the initial ordering of content boxes, as well as assignment of selection values to all components of the deduction graph (the notion of selection values was described earlier in this section).  *)
  method private create_DG =
    dedotter#process_dot "imps_dg.dot";
    let dimensions = dedotter#get_dimensions in
    print_string "Creating nodes"; flush stdout;
    let dummy name = function
	(x, y, w, h) -> 
	  print_string "."; flush stdout;
	  let (special, kind, text) = (!parser_dg#get_node_details name) in
	  let selection_name = (Hashtbl.length selection_names) + 1 in
	    Hashtbl.add selection_names name selection_name;
	  let ordering = (float((Hashtbl.length content_box_ordering) + 1) /. 10.) in
	    Hashtbl.add content_box_ordering name (ordering);
	  self#add_node 
	    name (special, kind) dimensions x y w h 
	    init_z text selection_name ordering
    in
      Hashtbl.iter dummy (dedotter#get_nodes);
    print_endline "done.";
    print_string "Creating arrows"; flush stdout;
    let dummy (tail, head) = function
	lst -> 
	  print_string "."; flush stdout;
	  self#add_arrow (self#get_node tail) (self#get_node head) lst
    in
      Hashtbl.iter dummy (dedotter#get_edges);
    print_endline "done.";
    ()

(** The argument of this method is a string that holds the name of particular node.  The return result of the method is a boolean value that is set to [true] when the supplied by the argument node is a part of a collapsed part of the graph.  *)
  method private is_it_in_collapsed name =
    let result = ref false in
    let dummy _ (_, list, _) =
      List.iter (
	fun (x, _, _) ->
	  if (String.compare name x = 0) then result := true; 
      ) list;
    in 
      Hashtbl.iter dummy collapsed;
      (!result)

(** This method displays the deduction graph on the screen.  When there is activity, such as dragging of objects on the screen, zooming or other visual movement, this method is called approximately 30 to 60 times per second (depending on how fast the graphical processing unit (GPU) is).   *)
  method private draw_DG =
    let dummy name _ = 
      if (self#is_it_in_collapsed name = false) then
	(self#get_node name)#rdraw (x1, y1, z1); 
      () 
    in 
      Hashtbl.iter dummy (dedotter#get_nodes);
      Hashtbl.iter dummy (collapsed);

    GlMisc.load_name (Hashtbl.find selection_names "arrow");

    let dummy (head, tail) _ = 
      if ((self#is_it_in_collapsed head = false) && (self#is_it_in_collapsed tail = false)) then
	(self#get_arrow head tail)#draw(); 
      () 
    in
      Hashtbl.iter dummy (dedotter#get_edges);

    let dummy (arrow, head, tail) =
      if ((self#is_it_in_collapsed head = false) && (self#is_it_in_collapsed tail = false)) then
	arrow#draw();
      ()
    in
      List.iter dummy added_arrows;
    ()

  (*========================================================*)
(** This variable holds a reference pointer to the instantiation of the TextGL class.  Only the reference to this object is kept here, because it is later passed and used by the all instances of the [Node] class to draw the text labels of each node.   *)
  val font = ref (new font)

(** This method instantiates a new object of the [Node] class.  Then the method includes the newly created object in all relevant data structures and processes of the current class.  *)
  method add_node name ((special:specialT), (kind:kindT)) 
    dimensions x y w h z text 
    selection_name ordering  =
    
    let (name_texture, _, _) = self#new_texture name in
    let new_node = new node (name, (special, kind), name_texture, 
			     (name ^ ":\\newline=====================\\newline" ^ text), 
			     dimensions, x, y, z, w, h, font, selection_name, ordering) in
    nodes <- nodes @ [(name, new_node)];
    ()

(** This method, similarly to the method described above, instantiates a new object of the [Arrow] class, and then includes it in all relevant data structures and processes of the current class.  *)
  method add_arrow from_Node to_Node pairs =
    let new_arrow = new arrow (from_Node, to_Node, pairs) in
    arrows <- arrows @ [new_arrow];
    ()

(** This method returns the reference pointer to an [Arrow] object.  This object represents the arrow that connects the supplied by the two arguments of the method nodes in the deduction graph. *)
  method get_arrow origin destination =
    let rec iterate = function
	arrow::rest -> 
	  if ((arrow#get_origin#get_name = origin
	      && arrow#get_destination#get_name = destination))
	  then arrow else iterate rest
      | [] -> raise (Error("No match of arrow from " ^ origin ^ " to " ^ "destination"))
    in
      iterate arrows

(** This method takes the string that is provided by the argument and creates a new OpenGL texture that can be used by OpenGL to display that string.  Also, the method loads the texture into the video memory in order to make it available for immediate use upon need.  *)
  method new_texture text =
    let texture = GlTex.gen_texture () in
    GlTex.bind_texture ~target:`texture_2d (texture);
    List.iter (GlTex.parameter ~target:`texture_2d)
      [ `border_color (1., 1., 1., 0.);
	`wrap_s `clamp;
	`wrap_t `clamp;
	`mag_filter `linear;
	`min_filter `linear ];
    let (image, w, h) = !font#get_text text in
    GluMisc.build_2d_mipmaps image;
    (texture, w, h)

(** This method retrieves the reference pointer to the object of class [Node] that is associated with the node represented by the string in the argument. *)
  method get_node name =
    let rec iterate = function
	(id, node)::rest -> if (id = name) then node else iterate rest
      | [] -> raise (Error("No match of node with this name: " ^ name))
    in iterate nodes

(** This method matches the integer that is supplied by the argument with a selection value (described earlier).  This is useful for finding the string name of the object that is assigned the selection value.  *)
  method get_hit_object_name hit_name =
    let answer = ref "" in
    let foo name = function n -> if (n = hit_name || n = hit_name mod 1000000) then answer := name in
      Hashtbl.iter foo selection_names;
      !answer  

  (* ========= SHOW PATH ============== *)

(** This list holds the string names of all nodes that participate in a visualization of a path between certain nodes in the deduction graph.  It is used by other methods as a reference on which parts of the graph must be highlighted.  *)
  val mutable path_flag_list = []	

(** This method resets (nullifies) the [path_flag_list] variable described above.  *)
  method private clear_path_hit =
    let foo name =
      (self#get_node name)#set_path_flag false
    in 
      List.iter foo path_flag_list;
      path_flag_list <- [];
      self#draw();
      ()

(** Based on the supplied by the argument selection value, which is possibly associated with some node, this method populates the [path_flag_list] that was specified above with all nodes that are traversed in order to reach the top node in the deduction graph (the main goal).  *)
  method show_back_path_hit hit_name =
    self#clear_path_hit;
    match (self#get_hit_object_name hit_name) with
	node when ((String.compare node "arrow" <> 0) && (String.length node > 0)) ->
	  begin
	    path_flag_list <- !parser_dg#find_the_way_back node;
	    let foo name =
	      (self#get_node name)#set_path_flag true
	    in 
	      List.iter foo path_flag_list;
	      self#draw();
	      ()
	  end
      | _ -> ()

(** Similarly to the [show_back_path_hit] method described above, this method populates the [path_flag_list] with all nodes that are traversed in order to reach all leaves in the graph by starting at the node associated with the supplied by the argument node selection value.  *)
  method show_forward_path_hit hit_name =
    self#clear_path_hit;
    match (self#get_hit_object_name hit_name) with
	node when ((String.compare node "arrow" <> 0) && (String.length node > 0)) ->
	  begin
	    path_flag_list <- !parser_dg#find_the_way_forward node;
	    let foo name =
	      (self#get_node name)#set_path_flag true
	    in 
	      List.iter foo path_flag_list;
	      self#draw();
	      ()
	  end
      | _ -> ()


  (* ========== END show path ==========*)


  (* ========== COLLAPSE ===============*)

(** In order to collapse a section of the deduction graph, the user needs to select two nodes that represent the endpoints of the part to be collapsed.  This variable holds the string names of these two nodes, which are stored in a tuple together with a boolean value that is [true] only when the user has finished choosing both endpoint nodes.  This positive truth value provides indication to the other methods that the collapsing procedure is ready to proceed upon request by the user.  *)
  val mutable collapse_selection = (false, "", "")

(** This method is called when the user makes his or her selection of the first node that represents an endpoint node in a yet to be executed collapsing procedure.  *)
  method begin_collapse_selection (hit_name:int) =
    let name = self#get_hit_object_name hit_name in
      if (String.compare name "" <> 0 && String.compare name "arrow" <> 0) then 
	begin
	  let (_, kind, _) = !parser_dg#get_node_details name in
	    if (kind = SQN) then 
	      begin
		let (_, _, two) = collapse_selection in
		let status = ref false in
		  if (String.length two > 0) then status := true;
		  collapse_selection <- (!status, name, two);
		  print_endline(name ^ " was selected as BEGIN collapse node.");
	      end
	    else print_endline "Collapsing can occur only between Sequent nodes!";
	end;
      ()

(** Similarly to the [begin_collapse_selection] method described above, this method is called when the user selects the second node that represents an endpoint in a yet to be executed collapsing procedure.  *)
  method end_collapse_selection (hit_name:int) =
    let name = self#get_hit_object_name hit_name in
      if (String.compare name "" <> 0 && String.compare name "arrow" <> 0) then 
	begin
	  let (_, kind, _) = !parser_dg#get_node_details name in
	    if (kind = SQN) then 
	      begin
		let (_, one, _) = collapse_selection in
		let status = ref false in
		  if (String.length one > 0) then status := true;
		  collapse_selection <- (!status, one, name);
		  print_endline(name ^ " was selected as END collapse node.");
	      end
	    else print_endline "Collapsing can occur only between Sequent nodes!";
	end;
    ()

(** This method is used only for debugging purposes.  It outputs in the command shell the names of the children of the node associated with the selection value provided by the argument.  *)
  method show_children (hit_name:int) =
    let name = self#get_hit_object_name hit_name in
      if (String.compare name "" <> 0 && String.compare name "arrow" <> 0) then 
	begin
	  let children = (!parser_dg#get_node_children name) in
	  print_endline (string_of_int(List.length children));
	  List.iter (
	    fun x -> print_endline x
	  ) children;
	end
      else ()

(** This variable holds the information associated with the ``temporary uncollapse procedure'' as defined in the Design chapter.  *)
  val mutable temporary_uncollapsed = (false, "","", 0., 0.)

(** This method executes the temporary uncollapse procedure upon the user's request.  To accomplish the procedure, the method relies on the information contained in the [temporary_uncollapsed] variable that is described above. *)
  method temporary_uncollapse (hit_name:int) =
    let name = self#get_hit_object_name hit_name in
      if (String.compare name "" <> 0 && String.compare name "arrow" <> 0) then 
	begin
	  let the_node = self#get_node name in
	  let xx, yy = the_node#get_x, the_node#get_y in
	  try
	    let (one, _, two) = Hashtbl.find collapsed name in
	      temporary_uncollapsed <- (true, one, two, xx, yy);
	      self#uncollapse hit_name;
	  with _ -> ()
	end;
    ()

(**/**)
  val mutable name_of_last_box = ""
(**/**)

(** This method voids the results achieved by the [temporary_uncollapse] method that is described above.  In other words, it is used to restore the collapsing effect of the parts of the graph that were initially collapsed (before calling the temporary uncollapse procedure).  *)
  method collapse_back_the_temporary_uncollapsed =
    let (status, one, two, xx, yy) = temporary_uncollapsed in
      if (status = true) then
	begin
	  self#collapse_procedure one two;
	  (self#get_node name_of_last_box)#reposition_node(xx, yy, 0.0);
	  temporary_uncollapsed <- (false, "", "", 0., 0.);
	  self#draw();
	end;
    ()

(** This method permanently uncollapses a collapsed node.  It completely restores the collapsed section of the deduction graph that was represented by the collapsed node associated with the supplied by the argument selection value.  *)
  method uncollapse (hit_name:int) =
    let name = self#get_hit_object_name hit_name in
      if (String.compare name "" <> 0 && String.compare name "arrow" <> 0) then 
	begin
	  try
	    (* recreating links in parser_dg *)
	    let (one, to_be_uncollapsed, two) = Hashtbl.find collapsed name in
	      (* first make them all exist *)
	      List.iter (
		fun (x, _, _) ->
		  let the_node = self#get_node x in
		  let kind = the_node#get_kind in
		    !parser_dg#add_new kind x Nil [];
	      ) to_be_uncollapsed;

	      (* then add all children and parents *)
	      List.iter (
		fun (x, children, parents) ->
		  let the_node = self#get_node x in
		  let kind = the_node#get_kind in
		  let opposite_kind = match kind with SQN -> INF | INF -> SQN in
		  List.iter (
		    fun child ->
		      !parser_dg#add_new_to_list kind x child;
		      !parser_dg#add_new_to_parent_list opposite_kind child x;
		      ()
		  ) children;
		  List.iter (
		    fun parent ->
		      !parser_dg#add_new_to_parent_list kind x parent;
		      !parser_dg#add_new_to_list opposite_kind parent x;
		      ()
		  ) parents;
	      ) to_be_uncollapsed;


	      (* removing all traces of the box *)
	      !parser_dg#remove_node name;
	      Hashtbl.remove collapsed name;
	      let dummy (arrow, head, tail) =
		if ((String.compare head name <> 0) && (String.compare tail name <> 0)) then true
		else false
	      in
		added_arrows <- List.filter dummy added_arrows;
		
		self#draw();
	  with _ -> ()
	end;
    ()

(**/**)
  val mutable counter = 0
(**/**)

(** This method executes a collapsing procedure based on the preliminary selection of beginning and end nodes (see above). It not only instantiates a new object of the [Node] class and configures it, but also creates the new arrows connecting it to the other parts of the deduction graph, includes the collapsed parts of the graph in a list of objects to be made invisible, and deals with all peculiarities and algorithms mentioned in the collapsing section of the Design chapter.  *)
  method private collapse_procedure one two =
    let path = !parser_dg#find_path_to_from one two in

      if ((List.length path) > 0) then
	begin
	  (*    let box_name = ("[BOX-" ^ string_of_int ((Hashtbl.length collapsed) + 1) ^ "]") in *)
	  counter <- counter + 1;
	  let box_name = ("[" ^ one ^ "==>" ^ two ^ "]-" ^ string_of_int(counter)) in
	    
	    name_of_last_box <- box_name;


	  let to_be_collapsed = ref path in

	  let inf_children = ref [] in

	  List.iter 
	    (fun x -> 
	       let (_, kind, _) = (!parser_dg#get_node_details x) in
		 (match kind with
		      SQN -> 
			begin
			  if (List.length (!parser_dg#get_node_children x) > 1) then
			    begin
			      let override_nodes = x :: (!parser_dg#find_path_to_from one x) in
			      to_be_collapsed := 
				List.filter 
				  (fun p -> if (List.mem p override_nodes) then false else true) 
				  !to_be_collapsed;
			    end;
			end
		    | INF -> 
			begin
			  let temp = 
			    List.filter 
			      (fun c -> if (List.mem c path) then false else true) 
			      (!parser_dg#get_node_children x) 
			  in
			    inf_children := !inf_children @ temp
			end
		 );
		 ()
	    ) 
	    path;
	  
	    let for_collapsing_full_info = ref [] in
	      List.iter (
		fun x ->
		  let parents = !parser_dg#get_node_parents x in
		  let children = !parser_dg#get_node_children x in
		  for_collapsing_full_info := !for_collapsing_full_info @ [(x, children, parents)]
	      ) !to_be_collapsed;

	    Hashtbl.add collapsed box_name (one, !for_collapsing_full_info, two);
	      
	    List.iter (fun x -> !parser_dg#remove_node x) !to_be_collapsed;
	    
	    !parser_dg#add_new INF box_name Box (!inf_children);
	    
	    !parser_dg#add_new_to_parent_list INF box_name one;
	    !parser_dg#add_new_to_parent_list SQN two box_name;
	    !parser_dg#add_new_to_list SQN one box_name;

	    List.iter 
	      (fun x -> 
		 !parser_dg#add_new_to_parent_list SQN x box_name
	      ) 
	      !inf_children;

	    let the_node = self#get_node (List.hd (path)) in
	      
	    let (dimensions, x, y, z, w, h, width_char, xx, yy, rx, ry, rz) = the_node#get_all_specs in
	      
	    let selection_name = (Hashtbl.length selection_names) + 1 in
	      Hashtbl.add selection_names box_name selection_name;
	      let ordering = (float((Hashtbl.length content_box_ordering) + 1) /. 10.) in
		Hashtbl.add content_box_ordering box_name (ordering);
		
		(* Create the collapsed text *)
		let collapsed_text = ref "" in
		let how_many = ref (List.length path) in
		let rec spaces x string character =
		  if (x = 0) then string
		  else spaces (x-1) (string ^ character) character
		in
		  List.iter (
		    fun x -> 
		      collapsed_text := (spaces !how_many "" " ") ^ "(" ^ x ^ "\\newline" ^ !collapsed_text;
		      how_many := !how_many - 1;
		  ) path;

		  collapsed_text := "(" ^ one ^ "\\newline" ^ !collapsed_text ^ 
		    (spaces ((List.length path)+1) "" " ") ^ "(" ^ two ^ (spaces ((List.length path)+2) "" ")");
		  
		  (* Add node to canvas space *)
		  self#add_node 
		    box_name (Box, INF) dimensions x y 
		    (width_char *. float_of_int(String.length box_name)) h 
		    (z) !collapsed_text selection_name ordering;
		  (self#get_node box_name)#rdraw(rx, ry, rz);
		  (self#get_node box_name)#reposition_node(xx, yy, 0.0);

		  
		  (* Create and add arrows *)
		  let new_arrow = new arrow ((self#get_node one), (self#get_node box_name), []) in
		    added_arrows <- added_arrows @ [(new_arrow, one, box_name)];
		    let new_arrow = new arrow ((self#get_node box_name), (self#get_node two), []) in
		      added_arrows <- added_arrows @ [(new_arrow, box_name, two)];

		      List.iter 
			(fun x -> 
			   let new_arrow = new arrow ((self#get_node box_name), (self#get_node x), []) in
			     added_arrows <- added_arrows @ [(new_arrow, box_name, x)];
			)
			!inf_children;
		      
		      self#draw();
		      collapse_selection <- (false, "", "");
	end
      else
	begin
	  print_endline "CANNOT COLLAPSE, NO DIRECT PATH BETWEEN SELECTED NODES!";
	end;
      ()

(** This variable contains a list tuples that represent the history of all collapsing procedures done so far in the process of exploring and manipulating the deduction graph in Panoptes.  Each tuple holds two strings: the first string is the beginning node and the second string is the ending node for each collapsing procedure that was ever done.  This way the comprehensive history of these operations is saved, so that when the user updates the graph with new incoming from IMPS information, Panoptes has a record of all collapsing procedures that need to be performed.  *)
  val mutable collapse_history = []

(** Similarly to the above variable, this variable keeps a comprehensive history of all occasions of user induced renaming of nodes in the deduction graph.  *)
  val mutable renaming_history = []

(** This method resets the history lists in the case of a global reset of Panoptes.  Such events usually occur when the user starts developing a brand new proof in IMPS.  Of course, this reset is requested by the user rather than happening automatically. *)
  method clear_history =
    collapse_history <- [];
    renaming_history <- [];
    ()

(** This method invokes the collapsing procedure.  The boolean value in the argument provides information to the system about whether the collapsing is a result of a direct request by the user, or it is an automatic history recall.  *)
  method execute_collapse is_manual =
    if (is_manual = true) then 
      begin
	let (status, one, two) = collapse_selection in
          if (status) then
	    begin
	      self#collapse_procedure one two;
	      collapse_history <- collapse_history @ [(one, two)];
	    end
	  else
	    begin
	      print_endline "SELECTION NOT MADE YET!";
	    end;
      end
    else
      begin
	List.iter (fun (one, two) -> self#collapse_procedure one two) collapse_history;
      end;
    ()

  (* ========== END collapse ===========*)


  (* =========== RENAMING NODES =============== *)

(** This method is called when the user initiates a renaming procedure on a certain node.  The string supplied by the first argument provides the old name (used by the method to identify the node to be renamed).  The string supplied by the second argument contains the new name of the node.  *)
  method rename_node old_name new_name =
    let the_node = self#get_node old_name in
    let (new_name_texture, _, _) = self#new_texture new_name in
      the_node#rename_node new_name_texture new_name (String.length new_name);
      renaming_history <- renaming_history @ [(old_name, new_name)];
      self#draw();
      ()

(** This method calls the [rename_node] method that is described above.  It first identifies the node associated with the selection value provided by the argument.  *)
  method renaming_procedure hit_name =
    let name = self#get_hit_object_name hit_name in
      if (String.compare name "" <> 0 && String.compare name "arrow" <> 0) then 
	begin
	  print_string ("Enter new name for the \"" ^ name  ^ "\" node: "); 
	  let new_name = read_line() in
	    self#rename_node name new_name;
	end
      else print_endline "Renaming is possible only on nodes of the graph!";
    ()

(**/**)
  method execute_renaming =
    List.iter (fun (old_name, new_name) -> self#rename_node old_name new_name) renaming_history;
    ()
(**/**)


  (* =========== end Renaming ================= *)

(** This variable holds a tuple of a few values.  The boolean value is set to [true] if the user is dragging a node at the present moment, while the other values represent the name of the dragged node and its position coordinates at the moment the dragging started.  *)
  val mutable dragged_node = (false, "", 0., 0., 0.)

(** This method is called each time the user presses down on the left mouse button.  Based on the selection value and click location supplied by the arguments, the method calls all appropriate methods in order to process this event.  *)
  method mouse_left_button hit_name posx posy =
    let name = self#get_hit_object_name hit_name in
      if (String.compare name "" <> 0 && String.compare name "arrow" <> 0) then 
	begin
	  let dummy coordinates=
	    let (p_x, p_y, _) = 
	      GluMat.project coordinates in
	    let pixel_depth = 
	      GlPix.read ~x:(int_of_float(p_x)) ~y:(int_of_float(p_y)) 
		~width:1 ~height:1 ~format:`depth_component ~kind:`float in    
	    let raw_pixel_depth = GlPix.to_raw pixel_depth in
	    let screen_z = Raw.get_float raw_pixel_depth ~pos:0 in
	    let offset_x = posx -. p_x
	    and offset_y = screenHeight -. posy -. p_y in
	      (screen_z, offset_x, offset_y)
	  in
	  let the_node = self#get_node name in
	    if (the_node#is_selected) then
	      begin
		let (screen_z, offset_x, offset_y) = 
		  dummy (the_node#get_box_x, the_node#get_box_y, the_node#get_box_z)
		in
		locked_node <- (true, name, screen_z, offset_x, offset_y);

		(* =============== CONTENT BOX ORDERING ====================== *)
		let order = (Hashtbl.find content_box_ordering name) in
		let foo node_name value =
		  if (value > order) then
		    begin
		      let new_value = value -. 0.1 in
		      Hashtbl.replace content_box_ordering node_name new_value;
		      (self#get_node node_name)#change_cont_ordering new_value;
		    end;
		in
		  Hashtbl.iter foo content_box_ordering;

		let new_order = (float(Hashtbl.length content_box_ordering) /. 10.) in
		Hashtbl.replace content_box_ordering name new_order;
		the_node#change_cont_ordering (new_order);
		self#draw();
		(* ================ END content box ordering ================= *)

	      end
	    else
	      begin
		let (screen_z, offset_x, offset_y) = 
		  dummy (the_node#get_x, the_node#get_y, the_node#get_z)
		in
		dragged_node <- (true, name, screen_z, offset_x, offset_y);
	      end;
	end
      else
	begin
	  locked_dg_drag <- (true, posx, screenHeight -. posy)
	end;
      ()

(** This method is called every time the user releases the left mouse button.  Basically, the method clears all variables holding information regarding certain actions, such as the [locked_node], the [locked_dg_drag], and the [dragged_node] variables, all of which were described above.  *)
  method mouse_left_button_release =
    locked_node <- (false, "", 0., 0., 0.);
    locked_dg_drag <- (false, 0., 0.);
    dragged_node <- (false, "", 0., 0., 0.);
    ()

(** This method is called when the user drags the mouse while holding down the left button.  *)
  method mouse_left_button_dragging (posx:float) (posy:float) =
    if (posx > 10.) && (posx < screenWidth -. 10.) && (posy > 10.) && (posy < screenHeight -. 10.) then 
      begin
	let (is_node_locked, name, screen_z, offset_x, offset_y) = locked_node in
	  if is_node_locked then 
	    begin
	      let posy = screenHeight -. posy in
	      let the_node = self#get_node name in
	      let (m_x, m_y, m_z) = GluMat.unproject (posx -. offset_x, posy -. offset_y, screen_z) in
		the_node#reposition_box (m_x, m_y, m_z); 
		Glut.postRedisplay();
	    end
	  else 
	    begin
	      let dummy last_x last_y posy =
		let diff_x = posx -. last_x
		and diff_y = posy -. last_y in
		let (p_x, p_y, p_z) = GluMat.project (x1, y1, z1 +. init_z) in
		let (n_x, n_y, n_z) = GluMat.unproject(p_x +. diff_x, p_y +. diff_y, p_z) in
		(n_x, n_y)
	      in

	      let (is_node_dragged, name, screen_z, offset_x, offset_y) = dragged_node in
		if is_node_dragged then
		  begin
		    let posy = screenHeight -. posy in
		    let the_node = self#get_node name in
		    let (m_x, m_y, m_z) = GluMat.unproject (posx -. offset_x, posy -. offset_y, screen_z) in
		      the_node#reposition_node (m_x, m_y, m_z); 
		      Glut.postRedisplay();
		  end
		else
		  begin
		    let (is_dg_dragged, last_x, last_y) = locked_dg_drag in
		      if (is_dg_dragged) then
			begin
			  let posy = screenHeight -. posy in
			  let (n_x, n_y) = dummy last_x last_y posy in
			    x1 <- n_x;
			    y1 <- n_y;
			    locked_dg_drag <- (true, posx, posy); 
			    Glut.postRedisplay();
			end;
		  end
	    end;
      end;
    ()

(** This method is called when the user presses the right mouse button over a certain object on the screen, which is identified by the selection value provided by the argument.  *)
  method mouse_right_button hit_name =
    if ((hit_name > 1) && (hit_name <= Hashtbl.length selection_names)) || (hit_name > 1000000) then
      begin
	let name = self#get_hit_object_name hit_name in
	  (self#get_node name)#toggle_selected();
	  self#draw();
      end;
    ()

(** This method is obsolete at the moment, but it used to provide functionality for highlighting the nodes when the mouse cursor hovers over them.  Subsequently, this feature of Panoptes was disabled as it severely degrades the performance of the system. *)
  method mouse_hover hit_name =
    let (is_locked, _, _, _, _) = locked_node in
    if (is_locked = false) then
      if (hit_name > 1) && (hit_name <= Hashtbl.length selection_names) then
	begin
	  let name = self#get_hit_object_name hit_name in
	  let the_node = self#get_node name in
	  the_node#hovered 1;
	    self#draw();
	    the_node#hovered 0
	end;
    ()

(** This method utilizes the OpenGL functionality to send an imaginary ray from a certain location (supplied by the arguments) in order to obtain the object the ray hits (the ray travels perpendicularly to the computer screen, away from the user).  The result is the selection value of the hit object, or a zero if the hit list is empty.  *)
  method scan_for_hits (xPos:float) (yPos:float) = 
    let selectBuf = Raw.create_static `uint 64 in
    GlMisc.select_buffer selectBuf;
    GlMat.mode `projection;
    GlMat.push();
    ignore(GlMisc.render_mode `select);
    GlMat.load_identity();
    GluMat.pick_matrix ~x:xPos ~y:(screenHeight -. yPos) ~width:1. ~height:1.;
    self#set_perspective;
    self#draw();
    let hits = GlMisc.render_mode `render in
    GlMat.mode `projection;
    GlMat.pop();
    GlMat.mode `modelview;
    (* ========= find the closest to the screen hit ================== *)
    match hits with 
	0 -> 0 
      | _ -> (
	  let closest_hit = ref 0 in
	  let closest_distance = ref 0 in
	    for i = 1 to hits do
	      let the_hit = Raw.get selectBuf (i * 4 - 1) in
	      let dist = Raw.get selectBuf (i * 4 - 3) in
		if (dist < !closest_distance) then
		  begin
		    closest_distance := dist;
		    closest_hit := the_hit;
		  end;
	    done;
	    !closest_hit)

(** This method resets all class variables.  It is also called during initialization when the class is first instantiated into an object.  *)
  method init () = 
    self#reset_all_data;
    GlClear.color (1., 1., 1.);
    Gl.enable `depth_test;
    Gl.enable `blend; 
    GlFunc.blend_func `src_alpha `one_minus_src_alpha;
    GlFunc.depth_func `less;
    GlPix.store (`unpack_alignment 1);
    GlTex.env (`mode `replace);
    GlDraw.shade_model `flat;
    !font#initialize();
    self#create_DG;
    self#draw();
    ()

(** This is a variable that holds a boolean value.  A [true] value indicates that the system should provide a notification message to the user regarding availability of new IMPS data, and prompt him or her to click the keystroke associated with the ``update'' function. *)
  val mutable new_data_available = false

(** This method occasionally checks to determine if there is new IMPS data available.  *)
  method new_data_available_message (is_it:bool) () =
    new_data_available <- is_it;
    ()

(** This method displays the message that notifies the user when new IMPS data is available. *)
  method draw_new_data_available_message () =
    GlMat.push();
    GlMat.load_identity();
    GlMat.translate ~x:(0.0) ~y:(12.) ~z:(init_z +. 500.) ();
    let message = "NEW DATA DETECTED, click UPDATE button!" in
    let ratio = !font#calc_ratio_block message in
    let x = 17. in
    GlMat.scale ~x:x ~y:(x /. ratio) ();
    !font#printGL message;
    GlDraw.color ~alpha:1.0 (1., 0., 0.);
    GlDraw.line_width (3.);
    GlDraw.begins `line_strip;
    GlDraw.vertex3 (-1.1, 1.1, 0.);
    GlDraw.vertex3 (-1.1, -1.1, 0.);
    GlDraw.vertex3 (1.1, -1.1, 0.); 
    GlDraw.vertex3 (1.1, 1.1, 0.);
    GlDraw.ends ();
    GlMat.pop();
    ()

(** This variable indicates the status of the help screen.  Its value is set to [true] if the help screen should be currently visible, and [false] otherwise.  *)
  val mutable help_text = false

(** This method toggle the status of the help screen between visible and invisible states.  *)
  method toggle_help_text =
    match help_text with
	true -> help_text <- false; ()
      | false -> help_text <- true; ()

(** This method displays the help screen.  *)
  method display_help_text =
    let help_message = 
      ("Q: quit program@" ^ 
	"F: fit to screen@" ^
	"D: display statistics@" ^
	"U: update graph@" ^
	"R: restart Panoptes@" ^
	"Z: zoom in@" ^
	"X: zoom out@" ^
	"T: show path to root@" ^
	"Y: show path to leaves@" ^
	"1 and 2: select start and@" ^
	"              end node for boxing@" ^
	"3 or 4: box or unbox@" ^
	"V: hold down to to unbox temporarily@" ^
	"/: rename node (look in console)@" ^
	"H: this message")
    in
      self#bitmap_text 30. 30. help_message;
      ()

(** This method redraws the whole frame.  It is called 30-60 times per second on average, which depends on how powerful the graphical card of the host machine is.  *)
  method draw () = 
    GlClear.clear [`color; `depth];
    if new_data_available then self#draw_new_data_available_message (); 
    GlMisc.init_names();
    GlMisc.push_name(0);

    self#draw_DG;

    if help_text then self#display_help_text;
    
    (*Gl.finish();*)
    Glut.swapBuffers();
    ()

(**/**)
  method private set_perspective =
    GluMat.perspective ~fovy:fovy ~aspect:(screenWidth /. screenHeight) ~z:(1., 4000.);
    ()
(**/**)

(** This method is called each time the Panoptes window is resized by the user.  It contains functionality to be used for integrating additional functionality in Panoptes to control multiple screens independently (see Future Work chapter).  *)
  method reshape ~w:w ~h:h = 
    let h = match h with 0 -> 1 | _ -> h in
    GlDraw.viewport ~x:0 ~y:0 ~w:w ~h:h;
    screenWidth <- float(w);
    screenHeight <- float(h);
    GlMat.mode `projection;
    GlMat.load_identity();
    self#set_perspective;
    GlMat.mode `modelview;
    Glut.postRedisplay();
    ()


  (* ================ SCENE ZOOMING ========================================= *)

(** This method performs a zoom on a content box.  The float number supplied by the argument is the chunk, by which the zoom should happen.  It can be negative for zooming in or positive for zooming out.  The integer number is the selection value of the content box to be zoomed on.  *)
  method zoom_content_box (chunk:float) (hit_name:int) () =
    let owner_node_name = self#get_hit_object_name (hit_name mod 1000000) in
    let the_node = self#get_node owner_node_name in
      the_node#scale_content_box chunk; 
      self#draw(); 
      ()

(** This method performs a zoom on the whole deduction graph by moving it further away or closer to the user in order to achieve zoom out or zoom in effect respectively.  *)
  method zoom chunk ((posx:int), (posy:int)) () =
    let posy = int_of_float(screenHeight) - posy in
    let _fovy = screenWidth *. fovy /. screenHeight in
    let alpha = (_fovy *. float(posx) /. screenWidth) -. (_fovy /. 2.) in
    let beta = (90. -. alpha) *. pi /. 180. in
    let diff = chunk /. tan(beta) in
      x1 <- x1 -. diff;
      let _fovy = fovy in
      let alpha = (_fovy *. float(posy) /. screenHeight) -. (_fovy /. 2.) in
      let beta = (90. -. alpha) *. pi /. 180. in
      let diff = chunk /. tan(beta) in
	y1 <- y1 -. diff;
	z1 <- z1 +. chunk; 
	self#draw();
	()


  (* =================================================================== *)

  method setOrthoProjection =
    GlMat.mode `projection;
    GlMat.push();
    GlMat.load_identity();
    GlMat.ortho ~x:(0., screenWidth) ~y:(0., screenHeight) ~z:(-1.,1.);
    GlMat.scale ~x:1. ~y:(-1.) ~z:1. ();
    GlMat.translate3 (0., -1. *. screenHeight, 0.);
    GlMat.mode `modelview;
    ()

  method resetPerspectiveProjection =
    GlMat.mode `projection;
    GlMat.pop();
    GlMat.mode `modelview;
    ()

  method bitmap_text x y text=
    self#setOrthoProjection;
    GlMat.push();
    GlMat.load_identity();
    GlDraw.color (1.,0.,0.);
    self#renderBitmapString x y 0. Glut.BITMAP_HELVETICA_18 text;
    GlMat.pop();
    self#resetPerspectiveProjection;
    ()


(** These methods are used as a supplement by other methods for achieving different supporting operations.  *)
  method renderBitmapString (x:float) (y:float) (space:float) (font) (str:string) =
    let length = String.length str in
    let x1, y1 = ref x, ref y in
      for i = 0 to length - 1 do
	if (Char.compare '@' str.[i] = 0) then
	  begin
	    x1 := x;
	    y1 := !y1 +. 20.;
	  end
	else
	  begin
	    GlPix.raster_pos ~x:!x1 ~y:!y1 ();
	    Glut.bitmapCharacter ~font:font ~c:(Char.code str.[i]);
	    x1 := !x1 +. float(Glut.bitmapWidth ~font:font ~c:(Char.code str.[i])) +. space;
	  end;
      done;
      ()

end;;
