(**   *)
(*
  Panoptes: An Exploration Tool for Formal Proofs
  Copyright (C) 2008  Orlin Grigorov

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

  To contact the author, email to: ogrigorov at gmail.com
*)


open Types
open TextGL

(** There is one instantiation of this class for each node in the deduction graph.  Referring to the source code and the names of the arguments, it can be understood what each of them means as the names of the arguments are meaningful.  *)
class node ((node_id:string), ((special:specialT), (kind:kindT)), node_texture, 
	    content_id, (dg_w, dg_h), pos_x, pos_y, init_z, width_original, height_original, font,
	    selection_name, (initial_ordering:float))=

  let screen_w = 800.
  and screen_h = 600. in

  let scale_x = if (dg_w >= dg_h) then screen_w /. dg_w else screen_h /. dg_h in
  let scale_y = scale_x in

  let calc_pos_x x = (x *. scale_x) -. screen_w /. 2.
  and calc_pos_y y = (y *. scale_y) -. screen_h /. 2. in

  let node_name_width = width_original *. scale_x /. 2.5 
  and node_name_height = height_original *. scale_y /. 2.5 in

  let width_per_character = node_name_width /. (float_of_int (String.length node_id)) in

object (self)

(** This method returns a tuple of values.  Each value represents a certain piece of information about the node.  The argument names, which can be found in the source code, are self-explanatory.  *)
  method get_all_specs = ((dg_w, dg_h), pos_x, pos_y, init_z, width_original, height_original,
			  (width_original /. (float_of_int (String.length node_id))),
			  self#get_x, self#get_y,
			  self#get_rx, self#get_ry, self#get_rz)

(** Returns the kind of the node represented by the particular object of this class.  The kind can be a sequent node or an inference node.  *)
  method get_kind = kind

(** A variable that represents a pointer to the OpenGL texture containing the display name of the node.  *)
  val mutable name_texture = node_texture

  val mutable x = calc_pos_x pos_x
  val mutable y = calc_pos_y pos_y
(** The x, y, and z coordinates of the initial layout location of the node.  *)
  val mutable z = init_z

  val mutable width = node_name_width
(** The width and the height of the node.  *)
  val mutable height = node_name_height

(** A string that contains the detailed information of the node.  It represents the contents of the associated with this node information box.  *)
  val mutable content_text = content_id

(** The ratio between the width and the height of the information box if it were to be drawn to the OpenGL window.  *)
  val mutable content_ratio = !font#calc_ratio_block content_id

(** This method renames the node and then calls all other appropriate methods that update the variables, which describe the content box (e.g.~the content box ratio, text, etc.  *)
  method rename_node new_name_texture new_name_text num_characters =
    name_texture <- new_name_texture;
    width <- (float_of_int (num_characters)) *. width_per_character;
    content_text <- new_name_text ^ " aka " ^ content_text;
    content_ratio <- !font#calc_ratio_block content_text;
    ()

  val mutable rx = 0.0
  val mutable ry = 0.0
(** The x, y, and z distances from the initial layout location of the node.  These can also be understood as the relative positions to the permanent position of the node.  This is used for implementing the ability to revert back to the initial layout of the deduction graph upon request by the user.  *)
  val mutable rz = 0.0

  method private get_rx = rx
  method private get_ry = ry
(** This method makes the previous three positions available to other parts of the system.  *)
  method private get_rz = rz
  
(** A flag that indicates whether the mouse cursor is hovering over the node at any particular moment.  *)
  val mutable hovered_flag = 0

(** A flag that indicates whether the content box is currently open (if it is visible).  *)
  val mutable selected_flag = 0

(** A flag that indicates whether the node participates in a path visualization procedure.  *)
  val mutable path_flagged = false

(** The initial and default width of the information box associated with the node.  *)
  val mutable cont_w = 100.

  val mutable center_box_x = 0.0
  val mutable center_box_y = 0.0
(** Coordinates of the current position of the content box.  *)
  val mutable center_box_z = init_z +. 50. +. initial_ordering

  method get_x = x +. rx 
  method get_y = y +. ry 
(** These three methods return the current global position of the node in the OpenGL window.  They take into account the initial layout position, as well as the relative position of the node.  *)
  method get_z = z +. rz

  method get_width = width
(** These methods return the width and the height of the node in OpenGL distance units.  *)
  method get_height = height

  method calc_x xx = ((calc_pos_x xx) +. rx)
(** These two methods convert the supplied by the arguments coordinates into global positions in respect to the current layout and node transformations (location translation, relative positions, scaling, etc.).   *)
  method calc_y yy = ((calc_pos_y yy) +. ry)

(** This method returns the string name of the node.  It can be further used for identification of the node in the other parts of the system and their data structures.  *)
  method get_name = node_id

  method get_box_x = center_box_x
  method get_box_y = center_box_y
(** These methods make the coordinates of the content box position available to the other parts of the system.  *)
  method get_box_z = center_box_z

  method get_path_flag = path_flagged
(** These two methods respectively return and set the flag that indicates whether the node participates in a path visualization procedure.  *)
  method set_path_flag status =
    path_flagged <- status;
    ()

(** This method draws (displays) the information content box of the node on the screen (if it is in a visible state).  *)
  method draw_content_box () =
    GlMat.push();

    GlMisc.load_name(selection_name + 1000000);

    (* ============== BOX ==============================*)
    GlMat.translate ~x:(center_box_x) ~y:(center_box_y) ~z:(center_box_z) ();
    GlMat.scale ~x:cont_w ~y:(cont_w /. content_ratio) ();

    !font#printGL_block content_text;

    GlDraw.color ~alpha:1.0 (1., 1., 1.);
    GlDraw.rect (-1.05, -1.05) (1.05, 1.05);

    GlDraw.color ~alpha:0.25 (0., 0., 0.);
    GlDraw.rect (-1.1, -1.1) (1.1, 1.1);


    GlMat.pop();
    GlMat.push();

    (*=============== LINK =============================*)

    GlDraw.color ~alpha:0.5 (0., 0., 0.);
    GlDraw.line_width (1.5);

    GlDraw.line_stipple (255);
    Gl.enable `line_stipple;

    GlDraw.begins `lines;
    GlDraw.vertex3 (self#get_x, self#get_y, self#get_z);
    GlDraw.vertex3 (center_box_x, center_box_y, center_box_z -. 1.);
    GlDraw.ends ();

    Gl.disable `line_stipple;

    GlDraw.color ~alpha:1.0 (0., 0., 0.);

    GlMat.pop();
    ()

(** This method can be used by other parts of the system to learn whether the content box of the node is currently open (visible).  *)
  method is_selected =
    match selected_flag with
	 1 -> true
       | _ -> false

(** This is the method that draws the node in the OpenGL drawing space.  *)
  method draw () =

    if (self#is_selected) then self#draw_content_box();

    GlMat.push();

    GlMisc.load_name(selection_name);

    GlMat.translate ~x:(self#get_x) ~y:(self#get_y) ~z:(self#get_z) ();
    GlMat.scale ~x:(width) ~y:(height) (); 

    (match hovered_flag with 
	 1 ->
	   begin
	     GlDraw.color ~alpha:1.0 (0., 0., 0.);
	     GlDraw.line_width (3.0);
	     GlDraw.begins `line_loop;
	     GlDraw.vertex3 (1., 1., 0.);
	     GlDraw.vertex3 (1., -1., 0.);
	     GlDraw.vertex3 (-1., -1., 0.);
	     GlDraw.vertex3 (-1., 1., 0.);
	     GlDraw.ends ();
	   end
       | _ -> ()
    );

    (match special with 
	 Nil -> () 
       | Box -> 
	   begin
	     (*GlDraw.line_stipple (255);
	     Gl.enable `line_stipple;*)
	     GlDraw.color ~alpha:1.0 (0.5, 0., 0.5);
	     GlDraw.line_width (3.0);
	     GlDraw.begins `line_loop;
	     GlDraw.vertex3 (1., 1., 0.);
	     GlDraw.vertex3 (1., -1., 0.);
	     GlDraw.vertex3 (-1., -1., 0.);
	     GlDraw.vertex3 (-1., 1., 0.);
	     GlDraw.ends ();
	     (*Gl.disable `line_stipple;*)
	     ()
	   end
       | Repeated -> 
	   begin
	     GlDraw.color ~alpha:1.0 (0.5, 0., 0.);
	     GlDraw.line_width (3.0);
	     GlDraw.begins `line_loop;
	     GlDraw.vertex3 (1., 1., 0.);
	     GlDraw.vertex3 (1., -1., 0.);
	     GlDraw.vertex3 (-1., -1., 0.);
	     GlDraw.vertex3 (-1., 1., 0.);
	     GlDraw.ends ();
	     ()
	   end
       | Grounded -> 
	   begin
	     GlDraw.color ~alpha:1.0 (0., 1., 0.);
	     GlDraw.line_width (3.0);
	     GlDraw.begins `line_loop;
	     GlDraw.vertex3 (1., 1., 0.);
	     GlDraw.vertex3 (1., -1., 0.);
	     GlDraw.vertex3 (-1., -1., 0.);
	     GlDraw.vertex3 (-1., 1., 0.);
	     GlDraw.ends ();
	     ()
	   end
    );

    Gl.enable `texture_2d;
    GlTex.bind_texture ~target:`texture_2d (name_texture);
    GlDraw.begins `quads;
    GlTex.coord2(0.0, 0.); GlDraw.vertex3 (1., 1., 0.);
    GlTex.coord2(0.0, 1.); GlDraw.vertex3 (1., -1., 0.);
    GlTex.coord2(1.0, 1.); GlDraw.vertex3 (-1., -1., 0.);
    GlTex.coord2(1.0, 0.); GlDraw.vertex3 (-1., 1., 0.);
    GlDraw.ends ();
    Gl.disable `texture_2d;

    GlMat.pop();
    ()

(** This method allows moving of the content box to a new position in the OpenGL drawing space.  *)
  method reposition_box (xx, yy, (_:float)) =
    center_box_x <- xx;
    center_box_y <- yy;
(*    self#draw();*)
    ()

(** This method allows moving of the node to a new position in the OpenGL drawing space.  *)
  method reposition_node (xx, yy, (_:float)) =
    x <- xx -. rx;
    y <- yy -. ry;
    self#draw();
    ()

(** Same as above, but instead of a new position, the arguments of this method represent values of the distances from the current position of the node to the original location of the node in the initial layout generated by GraphViz.  *)
  method rdraw (xx, yy, zz) =
    rx <- xx;
    ry <- yy;
    rz <- zz;
    self#draw();
    ()

(** Sets the flag when the mouse cursor is over the node and unsets it when mouse leaves.  *)
  method hovered flag =
    hovered_flag <- flag;
    ()

(** Toggle the flag that indicates whether the content box is open or closed.  *)
  method toggle_selected () =
    selected_flag <- (
      match selected_flag with 
	  0 -> 1 
	| 1 -> 0 
	| _ -> 0
    );
    ()

(** Scales the content box by a certain amount.  This creates the effect of zooming for the content box.  Notice that zooming of the content boxes is achieved via different method than the zooming performed on the whole graph:  the graph is zoomed by moving it closer or further away from the viewer.  *)
  method scale_content_box (amount:float) =
    cont_w <- 
      begin
	let new_size = cont_w +. amount in
	  match new_size with
	      a when a > 30. -> a
	    | _ -> 30.
      end;
    self#draw();
    ()

(** This method is used to pull the content box above all other content boxes of all other nodes.  This functionality is useful in the events when content boxes overlap and the user wants to focus on this particular content box.  Of course, the higher the order (supplied by the argument), the lower priority there is for this content box to be displayed, which is a machinery to facilitate the design of the [Canvas] class.  *)
  method change_cont_ordering new_order =
    center_box_z <- init_z +. 50. +. new_order;
    ()

end;;
