(** *)
(*
  Panoptes: An Exploration Tool for Formal Proofs
  Copyright (C) 2008  Orlin Grigorov

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

  To contact the author, email to: ogrigorov at gmail.com
*)


open Parser
open Canvas
open Sqn_grabber

(** A variable of type boolean.  A [true] value would run Panoptes in testing mode.  This mode is used for obtaining input from test files instead of IMPS. *)
let testing = true

(** The filename that contains the IMPS deduction graph. *)
let dg_filename = 
  match testing with
      true -> "test3.dg"
    | false -> ("/tmp/" ^ (Unix.getlogin()) ^ "-dg-file.dg")
;;

(** The filename that contains the detailed information about the sequent nodes in the deduction graph.  *)
let sqn_filename = 
  match testing with 
      true -> "test3.sqn"
    | false -> "/tmp/" ^ (Unix.getlogin()) ^ "-dg-file.sqn"
;;

(** An instance of the ``sqn_grabber'' class that is used for monitoring the changes occurring to the file that holds the detailed information of the sequent nodes.  *)
let grabber = ref (new sqn_grabber sqn_filename);;

(** An instance of the ``graph_data_class'' class that is used for parsing the IMPS  deduction graph file.  The class also creates the internal data structure to hold the deduction graph information. Also, it collects the detailed information of all sequent nodes. *)
let dg =  ref (new graph_data_class ((Sys.getcwd()), dg_filename, (sqn_filename ^ ".mod")));;

(** A constant that is used to regulate the rate of scaling when using that operation on an information box. *)
let content_box_scroll_chunk = ref 5.;;

(** A constant that is used to regulate the rate of zooming when using that operation on the graph.  *)
let scene_scroll_chunk = ref 25.;;

(** Initializes the OpenGL window and creates all keyboard and mouse event listeners.  This is the main ``loop'' of Panoptes. *)
let main_GLUT () =

  ignore(Glut.init Sys.argv);
(*  let (pin, pout) = Unix.pipe() in
    Unix.dup2 pout Unix.stderr;*)
    Glut.initDisplayString ~str:"rgba double depth>=24 stereo=0";

  Glut.initWindowSize ~w:800 ~h:600;


  ignore (Glut.createWindow ~title:("Panoptes [" ^ dg_filename ^ "]"));
(*  print_endline ("Graphics: " ^ GlMisc.get_string `renderer ^ 
		   " (version " ^ GlMisc.get_string `version ^ ")");
  print_endline ("Vendor: " ^ GlMisc.get_string `vendor);
  print_endline (
    let depth = Glut.get ~gtype:Glut.WINDOW_DEPTH_SIZE in
    "Depth buffer precision: " ^ string_of_int(depth) ^ " bits" ^
      (if (depth < 24) then " (WARNING: depth precision is less than 24 bits. Some features will not function!)\n"
       else "\n")
  );
*)

  !dg#update;

  let view = new view dg in

  let rec keyboardUp_callback ~key ~x ~y = 
    match Char.lowercase (char_of_int key) with 
	'v' -> 
	  view#collapse_back_the_temporary_uncollapsed;
	  Glut.ignoreKeyRepeat ~ignore:false;

      | _ -> ()
  in
    Glut.keyboardUpFunc ~cb:keyboardUp_callback;

  let rec keyboard_callback ~key ~x ~y = 
    match Char.lowercase (char_of_int key) with 
	'q' -> !grabber#stop(); exit 0 
      | 'v' -> 
	  Glut.ignoreKeyRepeat ~ignore:true;
	  let hit = view#scan_for_hits (float(x)) (float(y)) in
	    view#temporary_uncollapse hit
	  
      | 'f' -> view#fit_to_screen()
      | 'd' ->
	  let (number_sqn, number_inf, sqn_grounded, sqn_repeated) = !dg#graph_stats in
	    print_endline (
	        "\n" ^
		"1) Sequent Nodes:\t" ^ string_of_int(number_sqn) ^ "\n" ^
		"\ta) Grounded:\t" ^ string_of_int(sqn_grounded) ^ "\n" ^
		"\tb) Repeated:\t" ^ string_of_int(sqn_repeated) ^ "\n" ^
		"2) Inference Nodes:\t" ^ string_of_int(number_inf) ^
		"\n")
      | 'r' ->
	  begin
	    !grabber#stop();
	    grabber := new sqn_grabber sqn_filename;
	    dg := new graph_data_class ((Sys.getcwd()), dg_filename, (sqn_filename ^ ".mod"));
	    view#clear_history;
	    keyboard_callback ~key:(int_of_char 'u') ~x:0 ~y:0;
	    print_endline "Now restarting!";
	  end
      | 'u' -> 
	  begin
	    view#new_data_available_message false ();
	    !dg#update;
	    view#init();
	    view#execute_collapse false;
	    view#execute_renaming;
	  end
      | 'z' ->
	  begin (* temporary fix for Macs and mouse scroll wheel *)
	    let hit = view#scan_for_hits (float(x)) (float(y)) in
	      match hit with
		  target when target > 1000000 -> 
		    view#zoom_content_box (-1. *. !content_box_scroll_chunk) 
		      target ();
		| _ -> view#zoom (-1. *. !scene_scroll_chunk) (x, y) ()
	  end
      | 'x' ->  (* temporary fix for Macs and mouse scroll wheel *)
	  begin
	    let hit = view#scan_for_hits (float(x)) (float(y)) in
	      match hit with
		  target when target > 1000000 -> 
		    view#zoom_content_box !content_box_scroll_chunk target ();
		| _ -> view#zoom !scene_scroll_chunk (x, y) ()
	  end
      | 't' ->
	  let hit = view#scan_for_hits (float(x)) (float(y)) in
	    view#show_back_path_hit hit
      | 'y' ->
	  let hit = view#scan_for_hits (float(x)) (float(y)) in
	    view#show_forward_path_hit hit
      | '1' ->
	  let hit = view#scan_for_hits (float(x)) (float(y)) in
	    view#begin_collapse_selection hit
      | '2' ->
	  let hit = view#scan_for_hits (float(x)) (float(y)) in
	    view#end_collapse_selection hit
      | '3' ->
	  view#execute_collapse true
      | '4' ->
	  let hit = view#scan_for_hits (float(x)) (float(y)) in
	    view#uncollapse hit

      | '0' ->
	  let hit = view#scan_for_hits (float(x)) (float(y)) in
	    view#show_children hit

      | 'h' -> view#toggle_help_text; view#draw(); ()
      | '/' -> 
	  let hit = view#scan_for_hits (float(x)) (float(y)) in
	    view#renaming_procedure hit
      | _ -> ()
  in
    Glut.keyboardFunc ~cb:keyboard_callback;


  let rec timerFunction ~value =
    let result = !grabber#poll() in 
      if result then
	begin
	  view#new_data_available_message true ();
	  view#draw();
	end; 
      Glut.timerFunc ~ms:500 ~cb:timerFunction ~value:0
  in Glut.timerFunc ~ms:500 ~cb:timerFunction ~value:0;


  let mouse_button_left = ref false
  and mouse_button_right = ref false in

  let mouseFunc ~button ~state ~x ~y =
    let (posx, posy) = (float(x), float(y)) in
    let hit = view#scan_for_hits posx posy in
      match state, button with
	  Glut.DOWN, Glut.LEFT_BUTTON ->
	    mouse_button_left := true;
 	    view#mouse_left_button hit posx posy
	      
	| Glut.UP, Glut.LEFT_BUTTON ->
	    mouse_button_left := false;
	    view#mouse_left_button_release
	      
	| Glut.DOWN, Glut.RIGHT_BUTTON ->
	    mouse_button_right := true;
	    view#mouse_right_button hit
	      
	| Glut.UP, Glut.RIGHT_BUTTON ->
	    mouse_button_right := false

	| Glut.DOWN, Glut.MIDDLE_BUTTON ->
	    print_endline "MIDDLE"
		    
	| _, _ -> ()
  in 
    Glut.mouseFunc ~cb:mouseFunc;


  let motionFunc ~x ~y =
    if !mouse_button_left then view#mouse_left_button_dragging (float(x)) (float(y))
    else ()
  in
    Glut.motionFunc ~cb:motionFunc;

(*
  let passiveMotionFunc ~x ~y =
    let hit = view#scan_for_hits (float(x)) (float(y)) in
      view#mouse_hover hit;
    ()
  in
    Glut.passiveMotionFunc ~cb:passiveMotionFunc;
*) 

  Glut.reshapeFunc ~cb:view#reshape;
  Glut.displayFunc ~cb:view#draw;


  (*======================= MENU =======================*)
(*    let menuHandler ~value = 
      print_endline ("MENU item " ^ string_of_int(value));
      () 
    in
    ignore(Glut.createMenu ~cb:(menuHandler));
    Glut.addMenuEntry ~label:"Test 1" ~value:1;
    Glut.addMenuEntry ~label:"Test 2" ~value:2;
    Glut.addMenuEntry ~label:"Test 3" ~value:3;

    let menuStatusFunc ~status ~x ~y =
      print_endline "CHECK";
      (*menuStatus := (status, x, y);*)
      ()
    in
      Glut.menuStatusFunc ~cb:(menuStatusFunc);

    Glut.attachMenu ~button:Glut.MIDDLE_BUTTON; 
*)
    (*======================= end MENU ====================*)

 
  Glut.mainLoop()
;;


let _ =
  print_endline "";
  print_endline "\t/===============================\\";
  print_endline "\t| Panoptes: An Exploration Tool |"; 
  print_endline "\t|     for Formal Proofs         |";
  print_endline "\t|-------------------------------|";
  print_endline "\t|   Written by Orlin Grigorov   |";
  print_endline "\t|   McMaster University, 2008   |";
  print_endline "\t\\==============================/";
  print_endline "";

  main_GLUT()
;;
