(**   *)
(*
  Panoptes: An Exploration Tool for Formal Proofs
  Copyright (C) 2008  Orlin Grigorov

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

  To contact the author, email to: ogrigorov at gmail.com
*)


open Printf;;
open Genlex;;
open Types;;

(* =============================================== *)

(** The arguments for instantiation of this class consist of three strings.  The first argument provides the system path to the current directory, which is needed for dealing with a compatibility issue on some Mac systems. The second and third string provide the filenames for the IMPS deduction graph and the [sqn_grabber] storage files respectively. *)
class graph_data_class (path_cwd, dg_filename, sqn_filename) =
object(self)

(** This hash table holds the details for the information boxes of each sequent node.  The table is indexed by the text name of the sequent node in IMPS, and it contains a tuple of the following elements: one indicating whether the node is special (repeated, grounded, etc.) and one that holds a string of text that contains the details of the node.  The third element of the tuple is not used.  It is added for convenience and eventual future expansion of functionality.  *)
  val mutable sqn_data = Hashtbl.create 0

(** The elements of this hash table comprise the set of all sequent nodes in the deduction graph.  They are indexed by their IMPS names, and the tuple of information contains their specialty and two separate lists of strings:  one for all children of the node and one for all parents of the node.  *) 
  val mutable sqn_list = Hashtbl.create 0

(** Similarly to [sqn_list] this hash table stores all inference nodes in the deduction graph.  They are are indexed by their names in IMPS and their assigned numbers in the parser method of this class.   *)  
  val mutable inf_list = Hashtbl.create 0

(*----COLLAPSING ACCOMODATION------------------------------------------------------*)

(** This method removes the existence of the node indicated by the parameter [name].  After this method completes its operation, the indicated node does not appear in any of the data structures of this class.  *)
  method remove_node name =
    Hashtbl.remove sqn_list name;
    Hashtbl.remove inf_list name;

    let dummy x y table =
      (match y with
           (special, list, parent_list) ->
	     begin
	       let new_list = List.filter 
		 (fun a -> 
		    if (String.compare a name = 0) then false 
		    else true) list 
	       in
		 
	       let new_parent_list = List.filter 
		 (fun a -> 
		    if (String.compare a name = 0) then false 
		    else true) parent_list 
	       in
		 
		 Hashtbl.replace table x (special, new_list, new_parent_list);
	     end
      ); ()
    in
    Hashtbl.iter (fun x y -> dummy x y sqn_list) sqn_list;
    Hashtbl.iter (fun x y -> dummy x y inf_list) inf_list;
    ()



(*---------------------------------------------------------------------------------*)
(** This method returns four integers: the number of sequent nodes, the number of inference nodes, the number of grounded nodes, and the number of nodes that are repeated and form a cycle in the deduction graph.   *)
  method graph_stats =
    let number_sqn = Hashtbl.length sqn_list
    and number_inf = Hashtbl.length inf_list in
    let sqn_grounded = ref 0
    and sqn_repeated = ref 0 in
    let rec foo _ = function
	(special, _, _) -> 
	  (match special with
	      Nil -> ()
	    | Grounded -> sqn_grounded := !sqn_grounded + 1
	    | Repeated -> sqn_repeated := !sqn_repeated + 1
	    | Box -> ()
	  );
    in Hashtbl.iter foo sqn_list;
    (number_sqn - !sqn_repeated, number_inf, !sqn_grounded, !sqn_repeated)

(** This method returns a string that represents the deduction graph in a format that is recognizable by the Graphviz external software.  Recall that this software is used for generating the initial layout of the deduction graph in Panoptes. *)
  method private make_dot_string =
    let dot = ref ("digraph G {\n\tsize =\"4,4\";\n") in
    let foo name = function
	(_, children, _) -> 
	  begin
	    dot := !dot ^ "\t" ^ name ^ ";\n";
	    List.iter (fun child -> dot := !dot ^ "\t" ^ name ^ " -> " ^ child ^ ";\n") children;
	  end
    in 
      Hashtbl.iter foo sqn_list;
      Hashtbl.iter foo inf_list;
      dot := !dot ^ "}";
      (!dot)

(** This method groups the calls to all other methods of this class to achieve a ``reset and recollect'' operation regarding all information that describes the deduction graph.  *)
  method update = 
    Sys.chdir path_cwd;
    print_string "Parsing IMPS data"; flush stdout;
    sqn_data <- Hashtbl.create 0;
    sqn_list <- Hashtbl.create 0;
    inf_list <- Hashtbl.create 0;
    print_string "."; flush stdout;
    self#duplicate_SQN_file_no_formfeed sqn_filename; 
    print_string "."; flush stdout; 
    self#read_sqn_file (sqn_filename ^ ".mod");
    print_string "."; flush stdout;
    self#duplicate_DG_file_no_hyphens dg_filename;
    print_string "."; flush stdout;
    self#read_dg_file (dg_filename ^ ".mod");
    print_string "."; flush stdout;
    self#write_dot_file ("imps_dg.dot") (self#make_dot_string);
    print_endline "done.";
    ()



  (* =============== TRAVERSING ===================== *)
(** This method returns a list of strings.  These strings contain the names of all parents of the node indicated by the parameter of the call.  *) 
  method get_node_parents name =
    try
      match (Hashtbl.find sqn_list name) with (_, _, lst) -> lst
    with _ -> 
      try
	match (Hashtbl.find inf_list name) with (_, _, lst) -> lst
      with _ -> 
	print_endline ("NOT FOUND"); 
	[]

(** This method returns a list of node names in the form of strings.  These nodes comprise the path from the indicated by the parameter node, all the way to the root node in the deduction graph (the oldest sequent node in the graph, a.k.a.~the main goal).  *)
  method find_the_way_back start_node =
    let visits = ref [start_node] in
    let rec loop name =
      if not (List.mem name !visits) then
	begin
	  visits := !visits @ [name];
	  List.iter loop (self#get_node_parents name)
	end
    in List.iter loop (self#get_node_parents start_node);
    (!visits)

(** This method, similarly to [get_node_parents], returns a list of all children of the indicated by the parameter node.  *)
  method get_node_children name =
    try
      match (Hashtbl.find sqn_list name) with (_, lst, _) -> lst
    with _ -> 
      try
	match (Hashtbl.find inf_list name) with (_, lst, _) -> lst
      with _ -> 
	print_endline ("NOT FOUND"); 
	[]

(** This method returns a list of nodes.  These nodes represent all successors of the indicated by the parameter node.  *)	
  method find_the_way_forward start_node =
    let visits = ref [start_node] in
    let rec loop name =
      if not (List.mem name !visits) then
	begin
	  visits := !visits @ [name];
	  List.iter loop (self#get_node_children name)
	end
    in List.iter loop (self#get_node_children start_node);
    (!visits)

(** This method returns a list of nodes, which comprise the path between the two nodes indicated by the two arguments.  If there is not a direct path between these two nodes, then the resulting list will remain empty.  *)
  method find_path_to_from to_node start_node =
    let result = ref [] in
    let visits = ref [start_node] in
    let rec loop path name =
      if (String.compare name to_node = 0) then result := path;
      if not (List.mem name !visits) then
	begin
	  visits := !visits @ [name];
	  let path = path @ [name] in
	  List.iter (loop path) (self#get_node_parents name)
	end
    in List.iter (loop []) (self#get_node_parents start_node);
    (!result)

      
  (* ============== END traversing ======================= *)

(** This method accepts the name of the node as an argument, and returns a tuple consisting of three things: the specialty of the node (repeated, grounded, etc.), the kind of the node (sequent or inference node), and its detailed information (supplied in the form of a string).    *)
  method get_node_details name =
    try
      match (Hashtbl.find sqn_list name) with (special, _, _) -> 
	begin
	  try
	    match (Hashtbl.find sqn_data name) with 
		(_, text, _) -> (special, SQN, text)
	  with _ -> (special, SQN, "dummy")
	end
    with _ -> 
      try
	match (Hashtbl.find inf_list name) with (special, _, _) -> (special, INF, "dummy")
      with _ -> 
	print_endline ("NOT FOUND"); 
	(Nil, SQN, "dummy")

(** This method writes to the disk the result from the method mentioned above: [make_dot_string].  The resulting file on the disk will be used in a subsequent step as an input to the external program GraphViz that is used to generate the initial layout of the deduction graph in Panoptes.  *)
  method write_dot_file filename dot =
    let oc = open_out filename in    
      fprintf oc "%s\n" dot;      
      close_out oc; 
    ()

(** This private method replaces the letter sequence ``IMPS_SQN_'' with ``SQN_'' in the supplied by the argument string.   *)
  method private sqn_clean name = Str.global_replace (Str.regexp "IMPS_SQN_") "SQN_" name

(** This private method adds a number to the name of an inference node.  This number shows how many times the inference rule that is represented by that node has been used in the deduction graph until that point. *)  
  method private inf_clean name = 
    let temp_name = (Str.global_replace (Str.regexp "_") "_" name) in
    let rec dummy init_name new_name n =
      try
	match (Hashtbl.find inf_list new_name) with _ -> 
	  dummy init_name (init_name ^ "__" ^ string_of_int(n)) (n + 1) 
      with _ -> new_name
    in dummy temp_name (temp_name ^ "__1") 2


  (* ========PRETTY PRINTER FOR DG ============================== *)

(** This method is used only for debugging purposes.  It implements a textual pretty printer of the deduction graph, the result of which is very similar to the format used internally in the IMPS user interface.  It displays its output in the active shell window that was used to start Panoptes.  *)
  method print_dg = 
    let rec string_of_list = function
	x::xs -> x ^ " " ^ string_of_list xs
      | _ -> ""
    in

    let	sqn_to_string id = function
	    (special, list, _) -> 
	      let status = (match special with 
				Nil -> "Nil" 
			      | Grounded -> "grounded" 
			      | Repeated -> "repeated"
			      | Box -> "box") in
		print_endline (id ^ " is " ^ status ^ "[" ^ (string_of_list list) ^ "]\n\n")
    in
      Hashtbl.iter sqn_to_string sqn_list;
      print_endline "";
      Hashtbl.iter sqn_to_string inf_list
      

  (* =======LEXER and PARSER of DG FILE======================== *)
(** This method adds a new node to the deduction graph.  It can be considered polymorphic in the sense that it can be used for adding both kinds of nodes---sequent and inference nodes.  Also, if known, a list of children of the node to be added can be supplied.  If such list is not available at the time of calling this method, then the [list] argument can accept an empty list.  *)
  method add_new kind name special list =
    match kind with
	SQN -> 
	  begin
	    match Hashtbl.mem sqn_list name with
		false -> Hashtbl.add sqn_list name (special, list, [])
	      | true -> 
		  let (_, children, parents) = Hashtbl.find sqn_list name in
		    Hashtbl.replace sqn_list name (Repeated, children, parents)
	  end
      | INF -> 
	  begin
	    match Hashtbl.mem inf_list name with
		false -> Hashtbl.add inf_list name (special, list, [])
	      | true -> 
		  let (_, children, parents) = Hashtbl.find inf_list name in
		    Hashtbl.replace inf_list name (Repeated, children, parents)
	  end
	    
(** This method adds a parent node to the list of parents of the indicated by the argument node.  Again, the kind of the target node (the one, whose list of parents is to be updated), must be passed as an argument. *)
  method add_new_to_parent_list kind name new_parent =
    let table =
      match kind with
          SQN -> sqn_list
	| INF -> inf_list
    in
      match (Hashtbl.find table name) with
          (special, list, parent_list) ->
	    if (List.mem new_parent parent_list = false) then
              Hashtbl.replace table name (special, list, parent_list @ [new_parent]);
	    ()

(** Similarly to the method mentioned above ([add_new_to_parent_list]), this method adds a new child node to the list of children of the indicated by the argument node.  *)	    	      
  method add_new_to_list kind name new_link =
    let table =
      match kind with
          SQN -> sqn_list
	| INF -> inf_list
    in
      match (Hashtbl.find table name) with
          (special, list, parent_list) ->
	    if (List.mem new_link list = false) then
              Hashtbl.replace table name (special, list @ [new_link], parent_list);
	    ()
	    


(** This is a very lengthy method, whose purpose is to read the IMPS file that describes the structure of the deduction graph.  After that is done, it parses the information and with the help of the other methods builds and populates the internal data structure used for storing the deduction graph.  For the parsing part, a lexer is created at the beginning, which separates the input into individual tokens.  The parser then matches these tokens according to the rules of the deduction graph abstract syntax tree, and populates the structure recursively.  *)
  method private read_dg_file filename = 

    let switch = function
	SQN -> INF
      | INF -> SQN
    in

    let clean name = function
	SQN -> self#sqn_clean name
      | INF -> self#inf_clean name
    in

    let stack = ref [] in

    let push link parent =
      stack := (link, parent) :: !stack; ()
    in

    let pop () =
      match !stack with
	  a :: rest ->
	    stack := rest;
	    a
	| [] -> (false, "")
    in

    let file_dg_lexer = make_lexer ["("; ")"; "_see"; "above_"; "GROUNDED"] in

    let rec sqn (link, parent) () = parser
	[< 'Kwd "("; _ = sqn (link, parent) () >] -> ()
      | [< 'Kwd ")"; _ = inf (pop()) () >] -> ()
      | [< 'Ident name; cleaned_name = parse_special_and_add SQN (clean name SQN) link parent; 
	   _ = inf (true, cleaned_name) (push link parent) >] -> ()
      | [< >] -> ()

    and inf (link, parent) () = parser
	[< 'Kwd "("; _ = inf (link, parent) () >] -> ()
      | [< 'Kwd ")"; _ = sqn (pop ()) () >] -> ()
      | [< 'Ident name; cleaned_name = parse_special_and_add INF (clean name INF) link parent; 
	   _ = sqn (true, cleaned_name) (push link parent) >] -> ()
      | [< >] -> ()

    and parse_special_and_add kind cleaned_name link parent = parser
	[< 'Kwd "_see"; 'Kwd "above_" >] ->
	  self#add_new kind cleaned_name Repeated [];
	  if (link) then 
	    begin
	      self#add_new_to_list (switch kind) parent cleaned_name;
	      self#add_new_to_parent_list (kind) cleaned_name parent;
	    end;
	  cleaned_name
	   
      | [< 'Kwd "GROUNDED" >] ->
	  self#add_new kind cleaned_name Grounded [];
	  if (link) then 
	    begin
	      self#add_new_to_list (switch kind) parent cleaned_name;
	      self#add_new_to_parent_list (kind) cleaned_name parent;
	    end;
	  cleaned_name

      | [< >] ->
	  self#add_new kind cleaned_name Nil [];
	  if (link) then 
	    begin
	      self#add_new_to_list (switch kind) parent cleaned_name;
	      self#add_new_to_parent_list (kind) cleaned_name parent;
	    end;
	  cleaned_name

    in

    
    let parse_this = parser [< _ = sqn (false, "") () >] -> () in
      parse_this(file_dg_lexer(Stream.of_channel (open_in filename)))


  (* ========REMOVE HYPHENS FROM DG FILE================ *)
(** In order to protect the parser from confusion when a few known to be problematic special characters are present, all such characters are replaced by equivalent in their meaning other strings.  Then the file can be used safely by the lexer and parser in the [read_dg_file] method of this class.  *)
  method private duplicate_DG_file_no_hyphens filename =
    let input_channel = open_in filename in
    let output_channel = open_out (filename ^ ".mod") in
    let rec copy_file ic oc =
      try
	let line = input_line ic in
	let stripped_line = Str.global_replace (Str.regexp "-") "_" line in
	let stripped_line = Str.global_replace (Str.regexp "%") "_" stripped_line in
	let stripped_line = Str.global_replace (Str.regexp "\\^") "_power_" stripped_line in
	let stripped_line = Str.global_replace (Str.regexp "\\>") "_greater_" stripped_line in
	let stripped_line = Str.global_replace (Str.regexp "<=") "less_equal_" stripped_line in
	  fprintf oc "%s\n" stripped_line;
	  copy_file ic oc
      with e ->
	match e with
	    End_of_file -> close_in_noerr ic; close_out oc
	  | _ -> raise e
    in
      copy_file input_channel output_channel   

  (* ============================================================================*)
  (* ========PRETTY PRINTER FOR SQN============================== *)
(** This is a pretty printer for displaying the details of all sequent node in the command shell.  This method is used mainly for debugging purposes.  *)
  method print_sqn = 
    let	sqn_to_string id = function
	    (special, data, _) -> 
	      let status = (match special with 
				Nil -> "Nil" 
			      | Grounded -> "grounded" 
			      | Repeated -> "repeated"
			      | Box -> "box") in
		print_endline (id ^ " is " ^ status ^ " = \"" ^ data ^ "\"\n\n")
    in
      Hashtbl.iter sqn_to_string sqn_data
   
  (* ========REPLACE FORM FEED CHARS FROM SQN FILE================ *)
(** The file that contains the details of the sequent nodes is returned by IMPS in a special format.  Basically, IMPS uses the formfeed character to separate the information of the sequent nodes from each other, which confuses the parser.  Consequently, this method was created and used to replace all occurrences of this character in the file with another ``dummy'' character that is used to indicate the separation points.  *)
  method private duplicate_SQN_file_no_formfeed filename =
    let input_channel = open_in filename in
    let output_channel = open_out (filename ^ ".mod") in
    let rec copy_file ic oc =
      try
	let line = input_line ic in
	let stripped_line = Str.global_replace (Str.regexp "") "\n@\n\n" line in
	  fprintf oc "%s\n" stripped_line;
	  copy_file ic oc
      with e ->
	match e with
	    End_of_file -> 
	      close_in_noerr ic; 
	      fprintf oc "@";
	      close_out oc
	  | _ -> raise e
    in
      copy_file input_channel output_channel   
	

  (* =======LEXER and PARSER of SQN FILE======================== *)
(**/**)
  method private latex_format text =
    ("\\verb@" ^ text ^ "@\\newline")
(**/**)

(** This method creates a string from the file that holds the details of the sequent nodes.  Consequently, the method makes calls to the parser in an incremental manner by simultaneously populating the data structure (represented by the hash table [sqn_data]).  *)
  method private read_sqn_file filename =
    let get_id line = 
      let line = Str.global_replace (Str.regexp "^$") "" line in
	int_of_string(line)
    in

    let get_special line = match line with "nil" -> Nil | "t" -> Grounded | _ -> Nil in
    
    let rec get_body ic text =
      let line = input_line ic in
	if (String.compare line "@" = 0) then 
	  text
	else 
	  get_body ic (text ^ (self#latex_format line))
    in

    let rec get_block ic =
      let number = get_id (input_line ic) in
      let status = get_special (input_line ic) in
      let text = get_body ic "" in
	Hashtbl.add sqn_data ("SQN_" ^ string_of_int(number)) (status, text, []);
	let _ = input_line ic in
	get_block ic;
	()
    in

    let ic = open_in filename in
      try
	get_block ic;
	close_in ic;
        ()
      with e ->
	match e with
	    _ -> 
	      close_in_noerr ic;
	      ()



  (* =======OLD, might be useful at a later stage================ *)
(**/**)
  method private old_read_sqn_file filename =
    let file_sqn_lexer = make_lexer ["@"; "("; ","; ")"; ";"; "["; "]"] in
      
    let rec 
	parse_sqn_file = parser
	    [< 'Int number; 'Ident status; e = parse_sqn_body; e1 = parse_sqn_file >] -> 
	      let status2 = match status with "nil" -> Nil | "t" -> Grounded | _ -> Nil in
		Hashtbl.add sqn_data ("SQN_" ^ string_of_int(number)) (status2, e, []);
		e1
	  | [< >] -> ()
     
    and parse_sqn_body = parser
	[< 'Kwd "@" >] -> ""
      | [< 'Ident token; e = parse_sqn_body >] -> token ^ e
      | [< 'Kwd "("; e = parse_sqn_body >] -> "(" ^ e
      | [< 'Kwd ","; e = parse_sqn_body >] -> "," ^ e
      | [< 'Int n; e = parse_sqn_body >] -> (string_of_int n) ^ e
      | [< 'Kwd ")"; e = parse_sqn_body >] -> ")" ^ e
      | [< 'Kwd ";"; e = parse_sqn_body >] -> ";" ^ e
      | [< 'Kwd "["; e = parse_sqn_body >] -> "[" ^ e
      | [< 'Kwd "]"; e = parse_sqn_body >] -> "]" ^ e
      | [< >] -> ""
    in
      
    let parse_this = parser [< e = parse_sqn_file >] -> e in
      parse_this(file_sqn_lexer(Stream.of_channel (open_in filename)))
(**/**)

(* =============================================== *)



end;;



(* =============================================== *)

