(**   *)
(*
  Panoptes: An Exploration Tool for Formal Proofs
  Copyright (C) 2008  Orlin Grigorov

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

  To contact the author, email to: ogrigorov at gmail.com
*)


open Printf;;
open Genlex;;

(* =============================================== *)

(** This class provides the parser for the output generated by the layout program Graphviz. It invokes this program and then processes its result.  Contrary to the [Parser] module, which deals with parsing the IMPS ouput and builds the internal structure that represents the deduction graph, this module collects the information about the actual locations of all nodes and edges.    *)
class parser_dot =
object(self)
  
(** A variable that represents a hash table with all of the nodes (indexed by their string names).  Each record contains four numbers: the x and y positions, the width and the height of the respective node.  *)
  val mutable nodes = Hashtbl.create 0

(** A variable that represents a hash table with all of the edges.  Each edge is indexed by a tuple of two strings: the string names of the originating and destination nodes. Each record consists of a list of tuples of two real numbers.  Each tuple represents a control point of the position of the edge.  Thus, the first and last points will coincide with the positions of the two connected nodes, while the pairs in between represent control points that can be used for building Bezier curves.  *)
  val mutable edges = Hashtbl.create 0

(**/**)
  val mutable dimensions = (1.0, 1.0)
(**/**)


  method get_nodes = nodes
(** These two methods make the hash tables that keep the information about all nodes and edges of the deduction graph available to the other parts of the system.  *)
  method get_edges = edges

(**/**)
  method get_dimensions = dimensions
(**/**)

(** This method implements the parser for the ouput result, generated by the automatic layout generator Graphviz.  *)
  method private read_dot_file dot_filename =
    let file_dot_lexer = make_lexer ["graph"; "node"; "edge"; "stop"; "<"; ">"] in

    let rec parse_node_name = parser
	[< 'Kwd "<"; a = parse_node_name; 'Kwd ">" >] -> a  (* avoiding the bug in DOT v.2.14 *)
      |	[< 'Ident str >] -> str
      | [< 'Int i >] -> string_of_int(i)
    in

    let parse_node = parser
	[< name = parse_node_name; 
	   'Float x; 'Float y; 
	   'Float width; 'Float height;
	   _ = parse_node_name; 'Ident _; 'Ident _; 'Ident _; 'Ident _
	>] -> Hashtbl.add nodes name (x, y, width, height)
    in

    let add_point_edge tail head x y =
      Hashtbl.replace edges (tail, head) ((Hashtbl.find edges (tail, head)) @ [(x, y)])
    in

    let dummy = parser
	[< >] -> ()
    in

    let rec parse_points tail head n = parser
      [< 'Float x; 'Float y; _ = (
	   match n with 
	       1 -> add_point_edge tail head x y; dummy
	     | _ -> add_point_edge tail head x y; parse_points tail head (n - 1)
	 ) >] -> ()
    in

    let parse_edge = parser
	[< tail = parse_node_name; head = parse_node_name; 'Int n;
	   _ = (Hashtbl.add edges (tail, head) []; parse_points tail head n);
	   'Ident _; 'Ident _
	>] -> ()
    in
      
    let rec parse_dot_file = parser
	    [< 'Kwd "graph"; 'Float _; 'Float width; 'Float height; _ = parse_dot_file >] -> 
	      dimensions <- (width, height); ()
	  | [< 'Kwd "node"; _ = parse_node; _ = parse_dot_file >] -> ()
	  | [< 'Kwd "edge"; _ = parse_edge; _ = parse_dot_file >] -> ()
	  | [< 'Kwd "stop" >] -> ()
    in
    

    let parse_this = parser [< e = parse_dot_file >] -> e in
      parse_this(file_dot_lexer(Stream.of_channel (open_in dot_filename)))

(* ==================================================================*)

(** This method invokes the automatic layout generator Graphviz.  *)
  method process_dot file =
    print_string "Generating layout"; flush stdout;
    nodes <- Hashtbl.create 0;
    edges <- Hashtbl.create 0;
    print_string "."; flush stdout;
    if Sys.command 
      (sprintf "dot -Tplain %s > %s.mod" file file) <> 0 then
	invalid_arg "parser_dot (dot error)";
    print_string "."; flush stdout;
    self#read_dot_file (file ^ ".mod");
    print_string "."; flush stdout;
    Sys.remove (file ^ ".mod");
    print_endline "done.";
    ()


end;;


(* 
   dot -- lay out directed graphs
   neato -- counterpart of dot for undirected graphs
   twopi -- for radial graph layouts
   circo -- for circular graph layouts
   fdp -- another layout engine (really fits the stuff)
*)
