(**   *)
(*
  Panoptes: An Exploration Tool for Formal Proofs
  Copyright (C) 2008  Orlin Grigorov

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.

  To contact the author, email to: ogrigorov at gmail.com
*)


open Printf

class font =
(*
created with:
./ttf2png -o font.png -r32,127 -s40 -l128 -c33 -a courbd.ttf

*)

  let start_char_code = 32
  and finish_char_code = 127 in

  let current_directory = Sys.getcwd() in

object(self)
(** A hash table that is indexed by an integer that represents a character of the alphabet.  Each record contains a tuple, consisting of the graphical image of that character and its width and height in pixel units. *)  
  val mutable chars:((int, ([<`rgb], [<`ubyte]) GlPix.t * int * int) Hashtbl.t) = Hashtbl.create 0

(** A hash table that is indexed by an integer that represents a character of the alphabet.  Contrary to the hash table described above, this hash table does not keep an image but the pre-generated OpenGL texture for the character.  This is needed in order to skip the time consumed for generating a new OpenGL texture each time such texture is needed.  Such need usually occurs during synchronization of the deduction graph in Panoptes with the one in IMPS.  As stated many times already, speed performance in Panoptes is critical for the pleasant user experience. *)
  val mutable chars_texture = Hashtbl.create 0

(** This method initializes the font by creating two empty hash tables (as described above) that will contain the images and textures of all characters in the alphabet.  Then the method calls the [make_image] method (see below) that is used to populate these hash tables.  *)
  method initialize () =
    chars <- Hashtbl.create 0;
    chars_texture <- Hashtbl.create 0;
    self#make_image();
    ()

(** Opens a pregenerated binary image file that contains contains a sequence of all characters in the alphabet.  Then it loads it into an internal data structure that is returned along with its dimension in pixel units. *)
  method private input_binary c =
    let ic = open_in_bin (current_directory ^ "/font/character_" ^ string_of_int(c) ^ ".img") in
      try 
	let c_w = input_binary_int ic in
	let c_h = input_binary_int ic in
	let size = (in_channel_length ic) - 8 in
	let str = String.create size in
	  really_input ic str 0 size;
	  let raw = Raw.create `ubyte ~len:size in
	    Raw.sets_string raw ~pos:0 str;
	    let image = GlPix.of_raw raw ~format:`rgb ~width:c_w ~height:c_h in
	      flush stdout;            
	      close_in ic;
              (image, c_w, c_h)
      with e ->                    
	close_in_noerr ic;         
	raise e

  val mutable char_width = 0
(** These are two global variables that respectively hold the width and the height of the characters in the alphabet (measured in pixel units).  *)
  val mutable char_height = 0

(** Loads the font by traversing the data structure that is returned by [input_binary] method, and extracts each individual character from it. The data describing each character is then stored in the hash table [chars] for further use.  *)
  method private make_image () =
    print_string "Loading font"; flush stdout;
    for c = start_char_code to finish_char_code do 
      let (image, c_w, c_h) = self#input_binary c in
	self#add_texture c (image, c_w, c_h);
	Hashtbl.add chars c (image, c_w, c_h);
	print_char '.'; flush stdout;
    done;
    let (_, c_w, c_h) = Hashtbl.find chars start_char_code in
      char_width <- c_w;
      char_height <- c_h;
      print_endline "done.";
      ()

(** This is a public method that returns the pregenerated OpenGL texture of the requested by the argument character.  *)
  method get_char c  = Hashtbl.find chars c

(** Similarly to the method described above, this is a public method that generates and returns an OpenGL texture that represents the supplied by the argument string of characters.  *)
  method get_text (text:string) =
    let text = Str.global_replace (Str.regexp "_") "-" text in
    let length = String.length text in
    let image_w = (char_width) * length 
    and image_h = char_height
    in
    let image = GlPix.create `ubyte ~format:`rgb ~width:image_w ~height:image_h in
   
    for c = 0 to length-1 do
      let (char_image, _, _) = self#get_char (Char.code text.[length-1-c]) in
	for i = 0 to char_height - 1 do
	    Raw.sets (GlPix.to_raw image) ~pos:(3*(i*image_w + c*(char_width)))
	    (Raw.gets (GlPix.to_raw char_image) ~pos:(3*(i * char_width)) ~len:(3 * char_width))
	done;
    done;
    ((image, image_w, image_h) : ((([<`rgb], [<`ubyte]) GlPix.t) * int * int))  

(** This method is used to clean the supplied by the argument string from the characters used to represent new lines.  These characters are replaced by the symbol `@', which facilitates the formating of the text (shift new lines down and to the left when generating a new texture from a string of characters) done by other client methods. *)
  method private clean_text_newlines (text:string) =
    let text = Str.global_replace (Str.regexp (String.escaped "\\verb@")) "" text in 
    let text = Str.global_replace (Str.regexp (String.escaped "@")) "" text in 
    let text = Str.global_replace (Str.regexp (String.escaped "\\newline")) "@" text in 
    let text = Str.global_replace (Str.regexp "_") "-" text in
    let length = String.length text in
    let number_lines = ref 1 in
      for i = 0 to length-1 do
	if ((Char.compare text.[i] '@') = 0) then number_lines := !number_lines + 1
      done;
    (text, !number_lines)

(** This private method is used to load the texture that is associated by the argument into the video memory of the graphical card. This is extremely important as the hardware can then use it immediately when needed instead of sending queries to the main memory. This pre-loading of textures method proved to be very efficient through direct testing done with Panoptes.   *)
  method private add_texture code (image, w, h) =
    let texture = GlTex.gen_texture () in
    GlTex.bind_texture ~target:`texture_2d (texture);
    List.iter (GlTex.parameter ~target:`texture_2d)
      [ `border_color (1., 1., 1., 0.);
	`wrap_s `clamp;
	`wrap_t `clamp;
	`mag_filter `linear;
	`min_filter `linear ];
    GluMisc.build_2d_mipmaps image;
    Hashtbl.add chars_texture code (texture, w, h);
    ()


(** This method displays a line of text directly in the OpenGL drawing window (see [printGL_block] below for more detailed description of this process).  *)
  method printGL text =
    let (text, _) = self#clean_text_newlines text in
    let length = String.length text in
    let step_x = 2. /. float(length) in
      Gl.enable `texture_2d;      
      for i = 0 to length-1 do
	let (texture, w, h) = Hashtbl.find chars_texture (Char.code text.[i]) in
	  GlTex.bind_texture ~target:`texture_2d (texture);
	  GlDraw.begins `quads;
	  GlTex.coord2(0.0, 0.0); GlDraw.vertex3 (float(i+1) *. step_x -. 1., 1., 0.);
	  GlTex.coord2(0.0, 1.0); GlDraw.vertex3 (float(i+1) *. step_x -. 1., -1., 0.);
	  GlTex.coord2(1.0, 1.0); GlDraw.vertex3 (float(i) *. step_x -. 1., -1., 0.);
	  GlTex.coord2(1.0, 0.0); GlDraw.vertex3 (float(i) *. step_x -. 1., 1., 0.);
	  GlDraw.ends ();
	  ()
      done;
      Gl.disable `texture_2d;
    ()

(** This private method is used to preliminary analyze a block of text in order to determine the length of the longest line contained in it.  This is used to correctly determine the right amount of memory that needs to be reserved for the image that will eventually represent this block of text.  *)
  method private find_longest_line text =
    let longest = ref 0 in
    let length = String.length text in
    let tmp_len = ref 0 in
    for i = 0 to length-1 do
      if (Char.compare '@' text.[i] = 0) then
	begin
	  if (!tmp_len > !longest) then longest := !tmp_len;
	  tmp_len := 0;
	end
      else
	tmp_len := !tmp_len + 1;
    done;
    (!longest)

(** This method calculates the ratio between the width and the height of the block of text supplied by the argument.  For example, if the text consists of 4 lines, the longest of which contains 10 symbols, then the ratio will be {% $\frac{10}{4} = 2.5$. %} This is used by the objects of [Node] class in order to do the right calculation for proportionate scaling and positioning of the node and its components (the frame and the label). *)
  method calc_ratio_block (text:string) =
    let (text, lines) = self#clean_text_newlines text in
    let longest = ref (String.length text) in
    if (lines > 1) then longest := self#find_longest_line text;
    let content_ratio = 
      float(!longest * char_width) /. float(lines * (char_height + 15)) in
    content_ratio

(** This is a very popular public method as it is used extensively by other parts of the system.  It allows immediate display of the text provided by the argument in the OpenGL window.  Therefore, this method is not only responsible for calling the other private methods associated with this process (such as the method for cleaning the text from newlines and the method for generating a texture in case when it does not already exist), but also it is responsible for calling the OpenGL procedures to draw the text on the screen.  The reader should know that prior to calling this method, the position of the text to be displayed should be set in the OpenGL engine. *)
  method printGL_block text =
    let (text, lines) = self#clean_text_newlines text in
    let length = String.length text in
    let longest = ref length in
    if (lines > 1) then longest := self#find_longest_line text;
    let step_x = 2. /. float(!longest) in
    let step_y = 2. /. float(lines) in
    let current_line = ref (float(lines-1)) in
    let current_pos = ref (-1.0) in
    Gl.enable `texture_2d;      
      for i = 0 to length-1 do
	let (texture, w, h) = Hashtbl.find chars_texture (Char.code text.[i]) in
	  if (Char.compare '@' text.[i] = 0) then 
	    begin
	      current_line := !current_line -. 1.;
	      current_pos := -1.0;
	    end
	  else 
	    begin
	      current_pos := !current_pos +. 1.;

	      GlTex.bind_texture ~target:`texture_2d (texture);
	      GlDraw.begins `quads;
	      GlTex.coord2(0.0, 0.0); GlDraw.vertex3 ((!current_pos +. 1.) *. step_x -. 1., 
						      (!current_line +. 1.) *. step_y -. 1., 0.);

	      GlTex.coord2(0.0, 1.0); GlDraw.vertex3 ((!current_pos +. 1.) *. step_x -. 1., 
						      (!current_line) *. step_y -. 1., 0.);

	      GlTex.coord2(1.0, 1.0); GlDraw.vertex3 (!current_pos *. step_x -. 1., 
						      (!current_line) *. step_y -. 1., 0.);

	      GlTex.coord2(1.0, 0.0); GlDraw.vertex3 (!current_pos *. step_x -. 1.,
						      (!current_line +. 1.) *. step_y -. 1., 0.);
	      GlDraw.ends ();
	    end;
	  ()
      done;
      Gl.disable `texture_2d;
    ()

(** This method takes an ASCII code of a character (supplied by the argument), and then returns the index number that is associated with that character in the hash table that holds the pregenerated OpenGL textures. *)
  method get_texture code =
    Hashtbl.find chars_texture code

  initializer
    ()


end;;



