﻿module Value

open System
open Microsoft.FSharp.Quotations
open Quote

// Value type

type Value<'a> =
    | V of 'a
    | E of Expr<'a>

let Lift a = E <@ a @>

let IsV = function V _ -> true | _ -> false
let IsE = function V _ -> false | _ -> true

let GetV = function V a -> a | E a -> Eval a
let GetE = function V a -> <@ a @> | E a -> a


// Operators and constants

type Constant<'a>           = Value<'a>
type UnaryOp<'a,'b>         = Value<'a> -> Value<'b>
type BinaryOp<'a,'b,'c>     = Value<'a> -> Value<'b> -> Value<'c>
type TernaryOp<'a,'b,'c,'d> = Value<'a> -> Value<'b> -> Value<'c> -> Value<'d>

module Constant =
    let GenV v = V v
    let GenE ev = E ev
    let GetV c = GetV c
    let GetE c = GetE c

module UnaryOp =
    let Gen f ef = function
        | V v -> f v |> V
        | E e -> ef e |> E
    let AppV u v = v |> V |> u |> GetV
    let AppE u e = e |> E |> u |> GetE

    let Lift f uf = Gen (fun i -> f i) (fun i -> Quote.ReduceCall <@ (%uf) %i @>)
    let Flatten u = Lambda "i" (AppE u) |> E

module BinaryOp =
    let Gen f ef = fun x y ->
        match x,y with
        | V v1,V v2 -> f v1 v2 |> V
        | E e1,E e2 -> ef e1 e2 |> E
        | _ -> ef (GetE x) (GetE y) |> E
    let AppV b v w = b (V v) (V w) |> GetV
    let AppE b e f = b (E e) (E f) |> GetE

    let Lift f uf = Gen (fun i j -> f i j) (fun i j -> Quote.ReduceCall <@ (%uf) %i %j @>)
    let Flatten b = Quote.Lambda "i" (fun i -> Quote.Lambda "j" (fun j -> AppE b i j)) |> E


module TernaryOp =
    let Gen f ef = fun x y z ->
        match x,y,z with
        | V v1,V v2,V v3 -> f v1 v2 v3 |> V
        | E e1,E e2,E e3 -> ef e1 e2 e3 |> E
        | _ -> ef (GetE x) (GetE y) (GetE z) |> E
    let AppV t v w x = t (V v) (V w) (V x) |> GetV
    let AppE t e f g = t (E e) (E f) (E g) |> GetE

    let Lift f uf = Gen (fun i j k -> f i j k) (fun i j k -> Quote.ReduceCall <@ (%uf) %i %j %k @>)
    let Flatten t = Quote.Lambda "i" (fun i -> Quote.Lambda "j" (fun j -> Quote.Lambda "k" (fun k -> AppE t i j k))) |> E

let Unit = Constant.GenV ()
let Compare x y = BinaryOp.Lift compare <@compare@> x y
let Min x y = BinaryOp.Lift min <@min@> x y
let Max x y = BinaryOp.Lift max <@max@> x y

module Function =
    let Apply f x = BinaryOp.Gen (fun f x -> f x) (fun f x -> Quote.ReduceCall <@ (%f) %x @>) f x
    let Apply2 f x y = TernaryOp.Gen (fun f x y -> f x y) (fun f x y -> Quote.ReduceCall <@ (%f) %x %y @>) f x y
    let Apply3 f x y z = Quote.ReduceCall <@ (%GetE f) (%GetE x) (%GetE y) (%GetE z) @> |> E
    let Lambda n f = Quote.Lambda n (UnaryOp.AppE f) |> E
    let Lambda2 n1 n2 f = Quote.Lambda n1 (fun v1 -> Quote.Lambda n2 (fun v2 -> BinaryOp.AppE f v1 v2)) |> E

module Control =
    let Ignore a = UnaryOp.Lift ignore <@ignore@> a
    let Sequence a b = BinaryOp.Gen (fun a b -> a;b) (fun a b -> <@ %a; %b @>) a b
    let While c b = BinaryOp.Gen (fun c b -> while c do b done)
                                 (fun c b -> <@ while %c do %b done @>) c b
    let TryWith b w = BinaryOp.Gen (fun b w -> try b with _ -> w)
                                   (fun b w -> <@ try %b with _ -> %w @>) b w
    let TryFinally b f = BinaryOp.Gen (fun b f -> try b finally f)
                                      (fun b f -> <@ try %b finally %f @>) b f
    let If cd th el = TernaryOp.Gen (fun cd th el -> if cd then th else el)
                                    (fun cd th el -> <@ if %cd then %th else %el @>) cd th el

    let Let n a f = Quote.Let n (GetE a) (UnaryOp.AppE f) |> E
    let LetRec n a f = Quote.LetRec n (UnaryOp.AppE a) (UnaryOp.AppE f) |> E

module Tuple =
    let Fst x = UnaryOp.Lift fst <@fst@> x
    let Snd x = UnaryOp.Lift snd <@snd@> x

    let Pair a b = BinaryOp.Gen (fun a b -> a,b) (fun a b -> <@ %a,%b @>) a b

module Option =
    let Option = function
        | None -> V None
        | Some(V v) -> V(Some v)
        | Some(E e) -> E <@ Some %e @>
    let UnOption = function
        | V(Some v) -> V v
        | E e -> E <@ (%e).Value @>
        | _ -> failwith "Option pattern match failed"

    let Some x = UnaryOp.Lift Some <@Some@> x
    let IsSome x = UnaryOp.Gen (fun (x:_ option) -> x.IsSome) (fun x -> <@ (%x).IsSome @>) x
    //let Value x = UnaryOp.Gen (fun (x:_ option) -> x.Value) (fun x -> <@ (%x).Value @>) x

module Ref =
    let Ref c = UnaryOp.Lift ref <@ref@> c
    let Deref c = UnaryOp.Gen (!) (fun c -> <@ !(%c) @>) c
    let Assign c v = BinaryOp.Gen (fun c v -> c := v) (fun c v -> <@ %c := %v @>) c v
    let Incr c = UnaryOp.Lift incr <@incr@> c
    let Decr c = UnaryOp.Lift decr <@decr@> c

module Bool =
    let True = Constant.GenV true
    let False = Constant.GenV false

    let Not x = UnaryOp.Gen not (fun x -> <@ not %x @>) x
    let And x y = BinaryOp.Gen (&&) (fun x y -> <@ %x && %y @>) x y
    let Or x y = BinaryOp.Gen (||) (fun x y -> <@ %x || %y @>) x y

    let gt x y = BinaryOp.Gen (>) (fun x y -> <@ %x > %y @>) x y
    let lt x y = BinaryOp.Gen (<) (fun x y -> <@ %x < %y @>) x y
    let ge x y = BinaryOp.Gen (>=) (fun x y -> <@ %x >= %y @>) x y
    let le x y = BinaryOp.Gen (<=) (fun x y -> <@ %x <= %y @>) x y
    let eq x y = BinaryOp.Gen (=) (fun x y -> <@ %x = %y @>) x y
    let neq x y = BinaryOp.Gen (<>) (fun x y -> <@ %x <> %y @>) x y

    let (^=) = eq
    let (^<>) = neq
    let (^>) = gt
    let (^<) = lt
    let (^<=) = le
    let (^>=) = ge
    let (^&&) = And
    let (^||) = Or    

module Array =
    let Lift (v:Value<'a> seq) =
        let isv = ref true
        let res = [| for e in v do match e with V a -> yield a | _ -> isv := false |]
        if !isv then V res
        else Expr.NewArray(typeof<'a>, [for e in v -> (GetE e).Raw]) |> Expr.Cast |> E

module List =
    let Nth l i = BinaryOp.Gen (fun (l: _ list) i -> l.[i]) (fun l i -> <@ (%l).[%i] @>) l i
    let Cons c l = BinaryOp.Gen (fun c l -> c::l) (fun c l -> <@ %c :: %l @>) c l
    let IsEmpty l = UnaryOp.Gen (fun (l:_ list) -> l.IsEmpty) (fun l -> <@ (%l).IsEmpty @>) l
    let ToSeq l = UnaryOp.Lift List.toSeq <@List.toSeq@> l
    let Head l = UnaryOp.Lift List.head <@List.head@> l
    let Tail l = UnaryOp.Lift List.tail <@List.tail@> l
    let Length l = UnaryOp.Lift List.length <@List.length@> l
    let Reverse l = UnaryOp.Lift List.rev <@List.rev@> l

    let rec allPairs = function
        | [] -> []
        | h::t -> [for i in t -> (h,i)] @ allPairs t
    let setnth l i v = List.mapi (fun i' n -> if i' = i then v else n) l

    let Singleton x = UnaryOp.Gen (fun x -> [x]) (fun x -> <@ [%x] @>) x
    let Replicate n x = BinaryOp.Lift List.replicate <@List.replicate@> n x
    let Append x y = BinaryOp.Lift List.append <@List.append@> x y
    let Map f x = BinaryOp.Lift List.map <@List.map@> f x
    let Mapi f x = BinaryOp.Lift List.mapi <@List.mapi@> f x
    let Map2 f x y = TernaryOp.Lift List.map2 <@List.map2@> f x y
    let Fold f a x = TernaryOp.Lift List.fold <@List.fold@> f a x
    let Find f x = BinaryOp.Lift List.find <@List.find@> f x
    let FindIndex f x = BinaryOp.Lift List.findIndex <@List.findIndex@> f x
    let SetNth l i v = TernaryOp.Lift setnth <@setnth@> l i v
    let Exists f x = BinaryOp.Lift List.exists <@List.exists@> f x
    let Forall f x = BinaryOp.Lift List.forall <@List.forall@> f x

    open System.Collections.Generic
    let Empty() =  Constant.GenE <@ List() @>
    let New s = UnaryOp.Gen (fun (s:#seq<_>) -> List s) (fun s -> <@ List %s @>) s
    let Item l i = BinaryOp.Gen (fun (l:List<_>) i -> l.[i]) (fun l i -> <@ (%l).[%i] @>) l i
    let Count l = UnaryOp.Gen (fun (l:List<_>) -> l.Count) (fun l -> <@ (%l).Count @>) l
    let Add l i = BinaryOp.Gen (fun (l:List<_>) i -> l.Add i) (fun l i -> <@ (%l).Add %i @>) l i
    let Remove l i = BinaryOp.Gen (fun (l:List<_>) i -> l.Remove i) (fun l i -> <@ (%l).Remove %i @>) l i
    let RemoveAt l i = BinaryOp.Gen (fun (l:List<_>) i -> l.RemoveAt i) (fun l i -> <@ (%l).RemoveAt %i @>) l i
    let Contains l i = BinaryOp.Gen (fun (l:List<_>) i -> l.Contains i) (fun l i -> <@ (%l).Contains %i @>) l i

module Seq =
    let Empty() = Constant.GenE <@ Seq.empty @>
    let ToList s = UnaryOp.Lift Seq.toList <@Seq.toList@> s
    let CastTo s = UnaryOp.Gen (fun s -> s :> seq<_>) (fun s -> <@ %s :> seq<_> @>) s
    let Make a b = BinaryOp.Gen (fun a b -> {a..b}) (fun a b -> <@ {%a .. %b} @>) a b

    let Skip s i = BinaryOp.Lift Seq.skip <@Seq.skip@> s i
    let AllPairs s = UnaryOp.Gen (fun s -> List.allPairs(Seq.toList s) :> seq<_>)
                                 (fun s -> <@ List.allPairs(Seq.toList %s) :> seq<_> @>) s

    let IsEmpty f = UnaryOp.Lift Seq.isEmpty <@Seq.isEmpty@> f
    let Head f = UnaryOp.Lift Seq.head <@Seq.head@> f

    let Singleton x = UnaryOp.Gen (fun x -> seq[x]) (fun x -> <@ seq[%x] @>) x
    let Append x y = BinaryOp.Lift Seq.append <@Seq.append@> x y

    let Map f x = BinaryOp.Lift Seq.map <@Seq.map@> f x
    let Mapi f x = BinaryOp.Lift Seq.mapi <@Seq.mapi@> f x
    let Iter f x = BinaryOp.Lift Seq.iter <@Seq.iter@> f x
    let Fold f a x = TernaryOp.Lift Seq.fold <@Seq.fold@> f a x
    let Filter f x = BinaryOp.Lift Seq.filter <@Seq.filter@> f x
    let Exists f x = BinaryOp.Lift Seq.exists <@Seq.exists@> f x
    let Forall f x = BinaryOp.Lift Seq.forall <@Seq.forall@> f x
    let Find f x = BinaryOp.Lift Seq.find <@Seq.find@> f x
    let FindIndex f x = BinaryOp.Lift Seq.findIndex <@Seq.findIndex@> f x

module Dictionary =
    open System.Collections.Generic
    let New() = Constant.GenE <@ Dictionary() @>
    let Add d k v = TernaryOp.Gen (fun (d:Dictionary<_,_>) k v -> d.Add(k,v)) (fun d k v -> <@ (%d).Add(%k,%v) @>) d k v
    let Item d i = BinaryOp.Gen (fun (d:Dictionary<_,_>) i -> d.[i]) (fun d i -> <@ (%d).[%i] @>) d i

module SortedSet =
    open System.Collections.Generic
    let New c = UnaryOp.Gen (fun (c:IComparer<_>) -> SortedSet c) (fun c -> <@ SortedSet %c @>) c
    let Add s i = BinaryOp.Gen (fun (s:SortedSet<_>) i -> s.Add i) (fun s i -> <@ (%s).Add %i @>) s i
    let Remove s i = BinaryOp.Gen (fun (s:SortedSet<_>) i -> s.Remove i) (fun s i -> <@ (%s).Remove %i @>) s i
    let Count s = UnaryOp.Gen (fun (s:SortedSet<_>) -> s.Count) (fun s -> <@ (%s).Count @>) s
    let Min s = UnaryOp.Gen (fun (s:SortedSet<_>) -> s.Min) (fun s -> <@ (%s).Min @>) s
