﻿module Polynomial

open System
open Value
open Codegen
open Algebra
open Bool

type MonomialMonoid<'m> =
    inherit Monoid<'m>
    inherit Quotient<'m>

    abstract deg: UnaryOp<'m,int>
    abstract log: UnaryOp<'m,seq<int>>
    abstract vars: UnaryOp<'m,seq<int*int>>

    abstract fromLog: UnaryOp<seq<int>,'m>
    abstract fromVars: UnaryOp<seq<int*int>,'m>

type TermModule<'c,'m,'t> =
    inherit MonoModule<'c,'t>

    abstract CR: QuotientRing<'c>
    abstract MM: MonomialMonoid<'m>

    abstract div: BinaryOp<'t,'t,'t option>

    abstract c: UnaryOp<'t,'c>
    abstract m: UnaryOp<'t,'m>
    abstract make: BinaryOp<'c,'m,'t>

type PolynomialAlgebra<'c,'m,'t,'p> =
    inherit Order<'m>
    inherit Algebra<'c,'p>
    inherit Quotient<'p>

    abstract CR: QuotientRing<'c>
    abstract TM: TermModule<'c,'m,'t>

    abstract terms: UnaryOp<'p,seq<'t>>
    abstract deg: UnaryOp<'p,int>
    abstract isZero: UnaryOp<'p,bool>

    abstract LT: UnaryOp<'p,'t> // Leading Term
    abstract LM: UnaryOp<'p,'m> // Leading Monomial
    abstract LC: UnaryOp<'p,'c> // Leading Coefficient
    abstract RT: UnaryOp<'p,'p> // Remaining Terms

    abstract fromTerms: UnaryOp<seq<'t>,'p>
    abstract fromTerm: BinaryOp<'c,'m,'p>


module MonomialMonoid =
    let Gen(one,mul,deg,log,vars,mkLog,mkVars,div,gcd,lcm,τ) =
        let m = Monoid.Gen(one,mul)
        let q = Quotient.Gen(one,None,None,Some div,None,Some gcd,Some lcm,Some τ)
        {   new MonomialMonoid<_> with
            member x.zero = m.zero
            member x.add = m.add
            member x.deg = deg
            member x.log = log
            member x.vars = vars
            member x.fromLog = mkLog
            member x.fromVars = mkVars
            member x.longdiv = q.longdiv
            member x.div = q.div
            member x.rem = q.rem
            member x.gcd = q.gcd
            member x.lcm = q.lcm
            member x.τ = q.τ
        }

    let LogToVars lg = seq {
        let c = ref 0
        for e in lg do
            if e > 0 then yield !c,e
            incr c
    }

    let VarsToLog vs = seq {
        let n = ref 0
        for i,e in vs do
            while !n < i do
                incr n
                yield 0
            incr n
            yield e
    }

    let SortVars vs = Seq.sortBy fst <| seq { for _,e as v:int*int in vs do if e > 0 then yield v }

    let DenseMap2 f a b =
        let rec loggen = function
        | l,[] | [],l -> l
        | e1::l1,e2::l2 -> (f e1 e2)::loggen(l1,l2)
        loggen(Seq.toList a, Seq.toList b) |> Seq.ofList

    let SparseMap2 f a b =
        let rec vargen: _ -> (int*int)list = function
        | l,[] | [],l -> l
        | (i1,e1)::v1,(i2,e2)::v2 when i1 = i2 -> (i1,f e1 e2)::vargen(v1,v2)
        | (i1,_ as p1)::v1,((i2,_)::_ as n2) when i1 < i2 -> p1::vargen(v1,n2)
        | n1,p2::v2 -> p2::vargen(n1,v2)
        vargen(Seq.toList a, Seq.toList b) |> Seq.ofList

    let Dense =
        let map2 f = TernaryOp.Lift DenseMap2 <@ DenseMap2 @> (BinaryOp.Flatten f)

        Gen(V Seq.empty,
            map2 Idx.add,
            UnaryOp.Lift Seq.sum <@ Seq.sum @>,
            id,
            UnaryOp.Lift LogToVars <@ LogToVars @>,
            id,
            UnaryOp.Gen (fun a -> VarsToLog(SortVars a))
                        (fun a -> <@ VarsToLog(SortVars %a) @>),
            (fun m n -> codegen {
                use d = map2 Idx.sub m n
                let c = Seq.Exists (UnaryOp.Flatten <| Bool.gt Idx.zero) d
                return Control.If c (V None) (Option.Some d)
            }) |> Fun2 |> Generate |> Function.Apply2,
            map2 Min,
            map2 Max,
            map2 (fun a b -> Max Idx.zero (Idx.sub b a)))

    let Sparse =
        let map2 f = TernaryOp.Lift SparseMap2 <@ SparseMap2 @> (BinaryOp.Flatten f)

        Gen(V Seq.empty,
            map2 Idx.add,
            UnaryOp.Gen  (fun m -> Seq.sumBy snd m)
                         (fun m -> <@ Seq.sumBy snd %m @>),
            UnaryOp.Lift VarsToLog <@ VarsToLog @>,
            id,
            UnaryOp.Lift LogToVars <@ LogToVars @>,
            UnaryOp.Lift SortVars <@ SortVars @>,
            (fun m n -> codegen {
                use d = map2 Idx.sub m n
                let f = Tuple.Snd >> Bool.gt Idx.zero
                let c = Seq.Exists (UnaryOp.Flatten f) d
                return Control.If c (V None) (Option.Some d)
            }) |> Fun2 |> Generate |> Function.Apply2,
            map2 Min,
            map2 Max,
            map2 (fun a b -> Max Idx.zero (Idx.sub b a)))


module TermOrder =
    open UnaryOp

    module Dense =
        module internal c =
            let rec lex = function
            | [],[] -> 0
            | _,[] -> 1
            | [], _ -> -1
            | e1::l1,e2::l2 ->
                match compare e1 e2 with
                | 0 -> lex (l1,l2)
                | x -> x
            let rec revlex o = function
            | [],[] -> o
            | _,[] -> -1
            | [], _ -> 1
            | e1::l1,e2::l2 ->
                match compare e1 e2 with
                | 0 -> revlex o (l1,l2)
                | x -> revlex -x (l1,l2)

        let Lex (m:#MonomialMonoid<_>) =
            let cmp (a:seq<int>) b = c.lex (Seq.toList a, Seq.toList b)
            Order.Gen (fun a b -> Function.Apply2 (E<@cmp@>) (m.log a) (m.log b))
        let RevLex (m:#MonomialMonoid<_>) =
            let cmp (a:seq<int>) b = c.revlex 0 (Seq.toList a, Seq.toList b)
            Order.Gen (fun a b -> Function.Apply2 (E<@cmp@>) (m.log a) (m.log b))
        let DegLex (m:#MonomialMonoid<_>) =
            let cmp (d1:int,m1:seq<int>) (d2,m2) = match compare d1 d2 with 0 -> c.lex (Seq.toList m1, Seq.toList m2) | n -> n
            Order.Gen (fun a b -> Function.Apply2 (E<@cmp@>) (Tuple.Pair (m.deg a) (m.log a)) (Tuple.Pair (m.deg b) (m.log b)))
        let DegRevLex (m:#MonomialMonoid<_>) =
            let cmp (d1:int,m1:seq<int>) (d2,m2) = match compare d1 d2 with 0 -> c.revlex 0 (Seq.toList m1, Seq.toList m2) | n -> n
            Order.Gen (fun a b -> Function.Apply2 (E<@cmp@>) (Tuple.Pair (m.deg a) (m.log a)) (Tuple.Pair (m.deg b) (m.log b)))

    module Sparse =
        module internal c =
            let rec lex = function
            | [],[] -> 0
            | _,[] -> 1
            | [], _ -> -1
            | (i1,e1)::v1,(i2,e2)::v2 ->
                match compare i1 i2, compare e1 e2 with
                | 0,0 -> lex (v1,v2)
                | 0,x -> x
                | n,_ -> -n
            let rec revlex o = function
            | [],[] -> o
            | _,[] -> -1
            | [], _ -> 1
            | (i1,e1)::v1 as n1, ((i2,e2)::v2 as n2) ->
                match compare i1 i2, compare e1 e2 with
                | 0,0 -> revlex o (v1,v2)
                | 0,x -> revlex -x (v1,v2)
                | 1,_ -> revlex 1 (n1,v2)
                | _,_ -> revlex -1 (v1,n2)

        let Lex (m:#MonomialMonoid<_>) =
            let cmp (a:seq<int*int>) b = c.lex (Seq.toList a, Seq.toList b)
            Order.Gen (fun a b -> Function.Apply2 (E<@cmp@>) (m.vars a) (m.vars b))
        let RevLex (m:#MonomialMonoid<_>) =
            let cmp (a:seq<int*int>) b = c.revlex 0 (Seq.toList a, Seq.toList b)
            Order.Gen (fun a b -> Function.Apply2 (E<@cmp@>) (m.vars a) (m.vars b))
        let DegLex (m:#MonomialMonoid<_>) =
            let cmp (d1:int,m1:seq<int*int>) (d2,m2) = match compare d1 d2 with 0 -> c.lex (Seq.toList m1, Seq.toList m2) | n -> n
            Order.Gen (fun a b -> Function.Apply2 (E<@cmp@>) (Tuple.Pair (m.deg a) (m.vars a)) (Tuple.Pair (m.deg b) (m.vars b)))
        let DegRevLex (m:#MonomialMonoid<_>) =
            let cmp (d1:int,m1:seq<int*int>) (d2,m2) = match compare d1 d2 with 0 -> c.revlex 0 (Seq.toList m1, Seq.toList m2) | n -> n
            Order.Gen (fun a b -> Function.Apply2 (E<@cmp@>) (Tuple.Pair (m.deg a) (m.vars a)) (Tuple.Pair (m.deg b) (m.vars b)))


module TermModule =
    let Gen(cr:QuotientRing<_>,mm,one,mul,div,scalar,getc,getm,make) =
        let e = Monoid.Gen(one,mul)
        let m = MonoModule.Gen(e,cr,scalar)
        {   new TermModule<_,_,_> with
            member g.zero = m.zero
            member g.add = m.add
            member g.scalar = m.scalar
            member g.coefficients = cr :> Ring<_>
            member g.CR = cr
            member g.MM = mm
            member g.div = div
            member g.c = getc
            member g.m = getm
            member g.make = make
        }

    open Tuple
    open Option
    let Generic(cr,mm) =
        Gen(cr, mm,
            Pair cr.one mm.zero,
            (fun a b -> Pair (cr.mul (Fst a) (Fst b)) (mm.add (Snd a) (Snd b))),
            (fun a b -> Generate <| codegen {
                use c = cr.div (Fst a) (Fst b)
                use m = mm.div (Snd a) (Snd b)
                return! If ((IsSome c) ^&& (IsSome m))
                           (Return <| Some(Pair (UnOption c) (UnOption m)))
                           (Return <| Option None)
            }),
            (fun s t -> Pair (cr.mul s (Fst t)) (Snd t)),
            Fst, Snd, Pair)


module PolynomialAlgebra =
    let Gen(cr:QuotientRing<_>,tm,pr:QuotientRing<_>,embed,scalar,cmp,terms,deg,iszero,lt,lm,lc,rt,make,mkterm) =
        let a = Algebra.Gen(Homomorphism.Gen(cr:>Ring<_>,pr:>Ring<_>,embed),scalar)
        {   new PolynomialAlgebra<_,_,_,_> with
            member p.zero = a.zero
            member p.one = a.one
            member p.neg = a.neg
            member p.add = a.add
            member p.mul = a.mul
            member p.sub = a.sub
            member p.pow = a.pow
            member p.coefficients = a.coefficients
            member p.embedding = a.embedding
            member p.scalar = a.scalar
            member p.cmp = cmp
            member p.CR = cr
            member p.TM = tm
            member p.terms = terms
            member p.deg = match deg with Some deg -> deg | _ -> fun p -> tm.MM.deg(tm.m(lt p))
            member p.isZero = iszero
            member p.LT = lt
            member p.RT = rt
            member p.LM = match lm with Some lm -> lm | _ -> fun p -> tm.m(lt p)
            member p.LC = match lc with Some lc -> lc | _ -> fun p -> tm.c(lt p)
            member p.fromTerms = make
            member p.fromTerm = mkterm
            member p.longdiv = pr.longdiv
            member p.div = pr.div
            member p.rem = pr.rem
            member p.gcd = pr.gcd
            member p.lcm = pr.lcm
            member p.τ = pr.τ
        }

    let Generic(tm:TermModule<_,_,_>,o:Order<_>) =
        let app0,app1,app2 = GetV, UnaryOp.AppV, BinaryOp.AppV
        let getm,getc = app1 tm.m, app1 tm.c
        let sort = List.sortWith (fun a b -> - app2 o.cmp (getm a) (getm b))
        let rec collapse = function
            | [] -> []
            | x::y::ts when app2 o.cmp (getm x) (getm y) = 0 -> app2 tm.make (app2 tm.CR.add (getc x) (getc y)) (getm x) :: ts |> collapse
            | x::ts when getc x = app0 tm.CR.zero -> collapse ts
            | x::ts -> x :: collapse ts
        let coefmap f = List.map (fun t -> app2 tm.make (app1 f (getc t)) (getm t))
        let scalar s = [app2 tm.make s (app0 tm.MM.zero)] |> collapse
        let neg p = coefmap (tm.CR.neg) p
        let add p g = p @ g |> sort |> collapse
        let sub p g = add p (neg g)
        let mul (p:_ list) (g:_ list) = [for t in p do yield! [for s in g do yield app2 tm.add t s]] |> sort |> collapse
        let longdiv p g =
            let rec div x y =
                match x,y with
                | t,[] -> [],t
                | [],_ -> [],[]
                | m::ts,n::ss ->
                    match app2 tm.div m n with
                    | None -> [],x
                    | Some q ->
                        let d,r = div (sub ts (mul ss [q])) y
                        q::d,r
            let d,r = div p g
            d |> sort |> collapse, r
        let pr =
            QuotientRing.Gen(
                scalar (app0 tm.CR.zero) |> V,
                scalar (app0 tm.CR.one) |> V,
                UnaryOp.Lift neg <@neg@>,
                BinaryOp.Lift add <@add@>,
                BinaryOp.Lift mul <@mul@>,
                BinaryOp.Lift sub <@sub@> |> Some,
                None,
                BinaryOp.Lift longdiv <@longdiv@> |> Some,
                None, None, None, None, None)
        Gen(tm.CR, tm, pr,
            UnaryOp.Lift scalar <@scalar@>,
            None,
            o.cmp,
            List.ToSeq,
            Some (fun p -> Control.If (List.IsEmpty p) Idx.zero (tm.MM.deg (tm.m (List.Head p)))),
            List.IsEmpty,
            List.Head,
            None, None,
            List.Tail,
            UnaryOp.Gen (fun ts -> collapse(sort(Seq.toList ts))) (fun ts -> <@ collapse(sort(Seq.toList %ts)) @>),
            fun c m -> tm.make c m |> List.Singleton)
