﻿module Algebra

open Microsoft.FSharp.Math
open System.Numerics
open Value
open Codegen


// Morphisms

type Homomorphism<'s1,'s2,'o1,'o2> = {
    s: 'o1
    d: 'o2
    φ: UnaryOp<'s1,'s2>
}

type Endomorphism<'s,'o> = Homomorphism<'s,'s,'o,'o>


// Basic algebraic objects

type Order<'s> =
    abstract cmp: BinaryOp<'s,'s,int>


type Quotient<'s> =
    abstract longdiv: BinaryOp<'s,'s,'s*'s>
    abstract div: BinaryOp<'s,'s,'s option>
    abstract rem: BinaryOp<'s,'s,'s>
    abstract gcd: BinaryOp<'s,'s,'s>
    abstract lcm: BinaryOp<'s,'s,'s>
    abstract τ: BinaryOp<'s,'s,'s> // Crossfactor: τ i j * gcd i j = j; τ i j * i = lcm i j


type Monoid<'s> =
    abstract zero: Constant<'s>
    abstract add: BinaryOp<'s,'s,'s>

type Group<'s> =
    inherit Monoid<'s>
    abstract neg: UnaryOp<'s,'s>
    abstract sub: BinaryOp<'s,'s,'s>

type Ring<'s> =
    inherit Group<'s>
    abstract one: Constant<'s>
    abstract mul: BinaryOp<'s,'s,'s>
    abstract pow: BinaryOp<'s,int,'s>

type QuotientRing<'s> =
    inherit Ring<'s>
    inherit Quotient<'s>

type Field<'s> =
    inherit QuotientRing<'s>
    abstract div: BinaryOp<'s,'s,'s>
    abstract inv: UnaryOp<'s,'s>


// Composite algebraic objects

type MonoModule<'r,'m> =
    inherit Monoid<'m>
    abstract coefficients: Ring<'r>
    abstract scalar: BinaryOp<'r,'m,'m>

type Module<'r,'g> =
    inherit MonoModule<'r,'g>
    inherit Group<'g>

type Algebra<'r,'s> =
    inherit Module<'r,'s>
    inherit Ring<'s>
    abstract embedding: Homomorphism<'r,'s,Ring<'r>,Ring<'s>>


type Vectorspace<'r> =
    inherit Module<'r,Vector<'r>>
    abstract dim: int


// Implementation of morphisms

module Homomorphism =
    let Gen(s,d,φ) = {s=s; d=d; φ=φ}

module Endomorphism =
    let Gen(s,φ): Endomorphism<_,_> = Homomorphism.Gen(s,s,φ)


// Implementation of base algebraic objects

module Order =
    let Gen cmp =
        {   new Order<_> with
            member o.cmp = cmp
        }

    let GenType<'t when 't: comparison> =
        {   new Order<'t> with
            member o.cmp = BinaryOp.Gen compare (fun x y -> <@ compare %x %y @>)
        }


module Quotient =
    let Gen(zero,one,divrem,div,rem,gcd,lcm,τ) =
        {   new Quotient<_> with
            member r.longdiv = fun x y ->
                match x,y,divrem,one with
                | z,_,_,_ when z = zero -> Tuple.Pair zero zero
                | x,z,_,_ when z = zero -> Tuple.Pair zero x
                | x,o,_,Some one when o = one -> Tuple.Pair x zero
                | x,y,_,Some one when x = y -> Tuple.Pair one zero
                | x,y,Some dr,_ -> dr x y
                | x,_,_,_ -> Tuple.Pair zero x
            member r.div = fun x y ->
                match x,y,div,divrem,one with
                | z,_,_,_,_ when z = zero -> Option.Some zero
                | _,z,_,_,_ when z = zero -> Option.Some zero
                | x,o,_,_,Some one when o = one -> Option.Some x
                | x,y,_,_,Some one when x = y -> Option.Some one
                | x,y,Some d,_,_ -> d x y
                | x,y,_,Some dr,_ -> Option.Some(Tuple.Fst(dr x y))
                | _,_,_,_,_ -> Option.Some zero
            member r.rem = fun x y ->
                match x,y,rem,divrem,one with
                | z,_,_,_,_ when z = zero -> zero
                | x,z,_,_,_ when z = zero -> x
                | _,o,_,_,Some one when o = one -> zero
                | x,y,_,_,_ when x = y -> zero
                | x,y,Some r,_,_ -> r x y
                | x,y,_,Some dr,_ -> Tuple.Snd(dr x y)
                | x,_,_,_,_ -> x
            member m.gcd = match gcd with Some f -> f | _ -> fun _ _ -> zero
            member m.lcm = match lcm with Some f -> f | _ -> fun _ _ -> zero
            member m.τ = match τ with Some f -> f | _ -> fun _ _ -> zero
        }


module Monoid =
    type Homomorphism<'s,'t> = Homomorphism<'s,'t,Monoid<'s>,Monoid<'t>>

    let Gen(zero,add) =
        {   new Monoid<_> with
            member m.zero = zero
            member m.add  = fun x y ->
                match x,y with
                | z,a | a,z when z = zero -> a
                | _,_ -> add x y
        }

module Group =
    type Homomorphism<'s,'t> = Homomorphism<'s,'t,Group<'s>,Group<'t>>

    let Gen(zero,neg,add,sub) =
        let m = Monoid.Gen(zero,add)
        {   new Group<_> with
            member g.zero = m.zero
            member g.add  = m.add
            member g.neg  = function
                | i when i = zero -> zero
                | a -> neg a
            member g.sub  = fun x y ->
                match x,y,sub with
                | a,b,_ when a = b -> zero
                | a,z,_ when z = zero -> a
                | z,a,_ when z = zero -> neg a
                | x,y,None -> add x (neg y)
                | x,y,Some s -> s x y
        }

module Ring =
    type Homomorphism<'s,'t> = Homomorphism<'s,'t,Ring<'s>,Ring<'t>>

    let Gen(zero,one,neg,add,mul,sub,pow) =
        let rec mk_pow (one,mul) (x,n) =
            if n = 0 then one
            else if n = 1 then x
            else
                let y = mk_pow (one,mul) (x,n/2)
                let y = mul y y
                if n % 2 = 0 then y else mul x y

        let g = Group.Gen(zero,neg,add,sub)
        let m = Monoid.Gen(one,mul)
        {   new Ring<_> with
            member r.zero = g.zero
            member r.one  = m.zero
            member r.neg  = g.neg
            member r.add  = g.add
            member r.sub  = g.sub
            member r.mul  = fun x y ->
                match x,y with
                | z,_ | _,z when z = zero -> z
                | o,a | a,o when o = one -> a
                | _,_ -> mul x y
            member r.pow  = fun x y ->
                match x,y,pow with
                | z,_,_ when z = zero -> zero
                | o,_,_ when o = one -> one
                | x,y,None -> mk_pow (one,mul) (x, GetV y)
                | x,y,Some p -> p x y
        }

module QuotientRing =
    let Gen(zero,one,neg,add,mul,sub,pow,divrem,div,rem,gcd,lcm,τ) =
        let r = Ring.Gen(zero,one,neg,add,mul,sub,pow)
        let q = Quotient.Gen(zero,Some one,divrem,div,rem,gcd,lcm,τ)
        {   new QuotientRing<_> with
            member g.zero = r.zero
            member g.one  = r.one
            member g.neg  = r.neg
            member g.add  = r.add
            member g.sub  = r.sub
            member g.mul  = r.mul
            member g.pow  = r.pow
            member g.longdiv = q.longdiv
            member g.div = q.div
            member g.rem = q.rem
            member g.gcd = q.gcd
            member m.lcm = q.lcm
            member g.τ = q.τ
        }


    let MultiplicativeMonoid(r: Ring<_>) =
        Monoid.Gen(r.one,r.mul)

    let gcd a b =
        let rec ea a b = if b = 0 then a else ea b (a % b)
        ea (abs a) (abs b)

    let ZI = Gen(
                V 0,
                V 1,
                UnaryOp.Gen (fun x   -> - x)   (fun x   -> <@ - %x @>),
                BinaryOp.Gen(fun x y -> x + y) (fun x y -> <@ %x + %y @>),
                BinaryOp.Gen(fun x y -> x * y) (fun x y -> <@ %x * %y @>),
                BinaryOp.Gen(fun x y -> x - y) (fun x y -> <@ %x - %y @>) |> Some,
                None,
                None,
                BinaryOp.Gen(fun x y -> Some(x / y)) (fun x y -> <@ Some(%x / %y) @>) |> Some,
                BinaryOp.Gen(fun x y -> x % y) (fun x y -> <@ %x % %y @>) |> Some,
                BinaryOp.Gen(fun x y -> gcd x y) (fun x y -> <@ gcd %x %y @>) |> Some,
                BinaryOp.Gen(fun x y -> x * y / gcd x y) (fun x y -> <@ %x * %y / gcd %x %y @>) |> Some,
                BinaryOp.Gen(fun x y -> y / gcd x y) (fun x y -> <@ %y / gcd %x %y @>) |> Some)

    let ZZ = Gen(
                V 0I,
                V 1I,
                UnaryOp.Gen (fun x   -> - x)                 (fun x   -> <@ - %x @>),
                BinaryOp.Gen(fun x y -> x + y)               (fun x y -> <@ %x + %y @>),
                BinaryOp.Gen(fun x y -> x * y)               (fun x y -> <@ %x * %y @>),
                BinaryOp.Gen(fun x y -> x - y)               (fun x y -> <@ %x - %y @>) |> Some,
                BinaryOp.Gen(fun x y -> BigInteger.Pow(x,y)) (fun x y -> <@ BigInteger.Pow(%x,%y) @>) |> Some,
                None,
                BinaryOp.Gen(fun x y -> Some(x / y))         (fun x y -> <@ Some(%x / %y) @>) |> Some,
                BinaryOp.Gen(fun x y -> x % y)               (fun x y -> <@ %x % %y @>) |> Some,
                BinaryOp.Gen(fun x y -> BigInteger.GreatestCommonDivisor(x,y))
                            (fun x y -> <@ BigInteger.GreatestCommonDivisor(%x,%y) @>) |> Some,
                BinaryOp.Gen(fun x y -> x * y / BigInteger.GreatestCommonDivisor(x,y))
                            (fun x y -> <@ %x * %y / BigInteger.GreatestCommonDivisor(%x,%y) @>) |> Some,
                BinaryOp.Gen(fun x y -> y / BigInteger.GreatestCommonDivisor(x,y))
                            (fun x y -> <@ %y / BigInteger.GreatestCommonDivisor(%x,%y) @>) |> Some)

module Field =
    type Homomorphism<'s,'t> = Homomorphism<'s,'t,Field<'s>,Field<'t>>

    let Gen(zero,one,neg,add,mul,inv,sub,div,pow) =
        let div = fun x y ->
            match x,y,div with
            | a,z,_ when z = zero -> raise(System.DivideByZeroException())
            | z,a,_ when z = zero -> zero
            | a,o,_ when o = one -> a
            | o,a,_ when o = one -> inv a
            | a,b,_ when a = b -> one
            | x,y,None -> mul x (inv y)
            | x,y,Some d -> d x y
        let r =
            QuotientRing.Gen(
                zero,one,neg,add,mul,sub,pow,
                None,
                Some(fun x y -> Option.Some (div x y)),
                Some(fun _ _ -> zero),
                Some(fun _ _ -> one),
                Some mul,
                Some(fun _ x -> x))
        {   new Field<_> with
            member f.zero = r.zero
            member f.one  = r.one
            member f.neg  = r.neg
            member f.add  = r.add
            member f.mul  = r.mul
            member f.sub  = r.sub
            member f.div  = div
            member f.div  = r.div
            member f.rem  = r.rem
            member f.gcd  = r.gcd
            member f.lcm  = r.lcm
            member f.τ    = r.τ
            member f.longdiv = r.longdiv
            member f.inv  = function
                | z when z = zero -> raise(System.DivideByZeroException())
                | o when o = one -> one
                | a -> inv a
            member f.pow  = fun x y ->
                match pow with
                | Some p -> p x y
                | None ->
                    let y = GetV y
                    if y < 0 then f.inv(r.pow x (V -y))
                    else r.pow x (V y)
        }

    let MultiplicativeGroup(f: Field<_>) =
        Group.Gen(f.one,f.inv,f.mul,Some f.div)

    let Inverse(r:QuotientRing<_>) =
        match r with
        | :? Field<_> as f -> f.inv
        | _ -> fun x -> Option.UnOption(r.div r.one x)

    let QF = Gen(
                V 0.,
                V 1.,
                UnaryOp.Gen (fun x   -> - x)          (fun x   -> <@ - %x @>),
                BinaryOp.Gen(fun x y -> x + y)        (fun x y -> <@ %x + %y @>),
                BinaryOp.Gen(fun x y -> x * y)        (fun x y -> <@ %x * %y @>),
                UnaryOp.Gen (fun x   -> 1. / x)       (fun x   -> <@ 1. / %x @>),
                BinaryOp.Gen(fun x y -> x - y)        (fun x y -> <@ %x - %y @>) |> Some,
                BinaryOp.Gen(fun x y -> x / y)        (fun x y -> <@ %x / %y @>) |> Some,
                BinaryOp.Gen(fun x y -> x ** float y) (fun x y -> <@ %x ** float %y @>) |> Some)

    let QQ = Gen(
                V 0N,
                V 1N,
                UnaryOp.Gen (fun x   -> - x)              (fun x   -> <@ - %x @>),
                BinaryOp.Gen(fun x y -> x + y)            (fun x y -> <@ %x + %y @>),
                BinaryOp.Gen(fun x y -> x * y)            (fun x y -> <@ %x * %y @>),
                UnaryOp.Gen (fun x   -> 1N / x)           (fun x   -> <@ 1N / %x @>),
                BinaryOp.Gen(fun x y -> x - y)            (fun x y -> <@ %x - %y @>) |> Some,
                BinaryOp.Gen(fun x y -> x / y)            (fun x y -> <@ %x / %y @>) |> Some,
                BinaryOp.Gen(fun x y -> BigNum.PowN(x,y)) (fun x y -> <@ BigNum.PowN(%x,%y) @>) |> Some)

    let CC = Gen(
                V Complex.Zero,
                V Complex.One,
                UnaryOp.Gen (fun x   -> - x)                     (fun x   -> <@ - %x @>),
                BinaryOp.Gen(fun x y -> x + y)                   (fun x y -> <@ %x + %y @>),
                BinaryOp.Gen(fun x y -> x * y)                   (fun x y -> <@ %x * %y @>),
                UnaryOp.Gen (fun x   -> Complex.Reciprocal x)    (fun x   -> <@ Complex.Reciprocal %x @>),
                BinaryOp.Gen(fun x y -> x - y)                   (fun x y -> <@ %x - %y @>) |> Some,
                BinaryOp.Gen(fun x y -> x / y)                   (fun x y -> <@ %x / %y @>) |> Some,
                BinaryOp.Gen(fun x y -> Complex.Pow(x, float y)) (fun x y -> <@ Complex.Pow(%x, float %y) @>) |> Some)


// Implementation of module objects

module MonoModule =
    type Homomorphism<'r,'s,'t> = Homomorphism<'s,'t,MonoModule<'r,'s>,MonoModule<'r,'t>>

    let Gen(elems:Monoid<_>,coeffs,scalar) =
        {   new MonoModule<_,_> with
            member m.coefficients = coeffs
            member m.scalar = scalar
            member m.zero = elems.zero
            member m.add  = elems.add
        }

module Module =
    type Homomorphism<'r,'s,'t> = Homomorphism<'s,'t,Module<'r,'s>,Module<'r,'t>>

    let Gen(elems:Group<_>,coeffs,scalar) =
        {   new Module<_,_> with
            member m.coefficients = coeffs
            member m.scalar = scalar
            member m.zero = elems.zero
            member m.add  = elems.add
            member m.neg  = elems.neg
            member m.sub  = elems.sub
        }
   
module Algebra =
    type Homomorphism<'r,'s,'t> = Homomorphism<'s,'t,Algebra<'r,'s>,Algebra<'r,'t>>

    let Gen(embed:Ring.Homomorphism<'s,'t>,scalar) =
        let coef = embed.s
        let elem = embed.d
        {   new Algebra<_,_> with
            member a.zero = elem.zero
            member a.one  = elem.one
            member a.neg  = elem.neg
            member a.add  = elem.add
            member a.mul  = elem.mul
            member a.sub  = elem.sub
            member a.pow  = elem.pow
            member a.coefficients = coef
            member a.embedding = Homomorphism.Gen(coef,elem,embed.φ)
            member a.scalar = fun x y ->
                match scalar with
                | Some s -> s x y
                | None -> a.mul (embed.φ x) y
        }

    let FromRing(r:Ring<_>) = Gen(Homomorphism.Gen(r,r,id),None)

    let ZZ = FromRing QuotientRing.ZZ
    let ZI = FromRing QuotientRing.ZI
    let QF = FromRing Field.QF
    let QQ = FromRing Field.QQ
    let CC = FromRing Field.CC

module Vectorspace =
    open Vector.Generic
    let Gen(ring:Ring<'r>,dim) =
        let scalar =
            BinaryOp.Gen
                (fun s v -> map (fun x -> BinaryOp.AppV ring.mul s x) v)
                (fun s v -> <@ map (fun x -> %BinaryOp.AppE ring.mul %s <@x@>) %v @>)
        let elems =
            Group.Gen(
                UnaryOp.Gen (create dim)       (fun z   -> <@ create dim %z @>) ring.zero,
                UnaryOp.Gen (fun x   -> - x)   (fun x   -> <@ - %x @>),
                BinaryOp.Gen(fun x y -> x + y) (fun x y -> <@ %x + %y @>),
                None)
        {   new Vectorspace<_> with
            member v.coefficients = ring
            member v.scalar = scalar
            member v.dim = dim
            member v.zero = elems.zero
            member v.add  = elems.add
            member v.neg  = elems.neg
            member v.sub  = elems.sub
        }


let Idx = QuotientRing.ZI
