﻿module Codegen

open System
open Value

// Computation expressions with state and continuation passing style
type StateCPS<'state, 'value, 'answer> = 'state -> ('state -> 'value -> 'answer) -> 'answer
type SCBuilder() =
    member x.Return a : StateCPS<_,_,_> = fun s k -> k s a
    member x.Bind (m: StateCPS<_,_,_>, f: _ -> StateCPS<_,_,_>) : StateCPS<_,_,_> =
        fun s k -> m s (fun s' k' -> f k' s' k)
let sc = SCBuilder()

let k0 _ = id
let RunM m s0 = m s0 k0

let Fetch s k = k s s
let Store v _ k = k v ()


// Open record type
type State = list<string*obj>
type StateRecord<'a>(name:string) =
    member ip.Name = name

    member ip.Contains (s:State) =
        (List.tryFind (fun e -> fst e = name) s).IsSome
    member ip.Find (s:State) def : 'a =
        match List.tryFind (fun e -> fst e = name) s with
        | None -> match def with
                  | None -> failwith <| "Failed to locate field " + name + " of an open record."
                  | Some d -> d
        | Some a -> unbox(snd a)
    member ip.Store (v:'a) (s:State) : State =
        match List.tryFind (fun e -> fst e = name) s with
        | Some _ -> failwith <| "Field " + name + " of an open record is already present."
        | _ -> (name,box v)::s
    member ip.Remove (s:State) : State = List.filter (fun e -> fst e <> name) s
    member ip.Modify v s = ip.Store v <| ip.Remove s

    member ip.HasElement() = sc {
        let! s = Fetch
        return ip.Contains s
    }
    
    member ip.Lookup() = sc {
        let! s = Fetch
        return ip.Find s None
    }

    member ip.Lookup d = sc {
        let! s = Fetch
        return ip.Find s (Some d)
    }

    member ip.Extend v = sc {
        let! s = Fetch
        do! Store (ip.Store v s)
    }

    member ip.Exclude() = sc {
        let! s = Fetch
        do! Store (ip.Remove s)
    }

    member ip.Replace v = sc {
        let! s = Fetch
        do! Store (ip.Modify v s)
    }


// Variable counters
let GetNewVar(n,s) =
    let cnt = StateRecord("vars_" + n)
    let cv = cnt.Find s (Some <| ref 1)
    let n = n + "_" + (!cv).ToString()
    incr cv
    let s = cnt.Modify cv s
    n,s

// Code generation StateCPS monad
type StateGen<'w> = StateCPS<State,unit,'w>
type CodeGen<'v,'w> = StateCPS<State,Value<'v>,Value<'w>>
let Generate m = RunM m []

// Monadic code combinators
let Return a = fun s k -> k s a
let Let a = fun s k -> let n,s' = GetNewVar("v",s) in Control.Let n (a s' k0) (k s')
let Ignore a = fun s k -> k s (Control.Ignore (a s k0))
let Sequence a b = fun s k -> k s (Control.Sequence (a s k0) (b s k0))
let PrependV v a = fun s k -> Control.Sequence (a s k0) (k s v)
let Prepend a = fun s k -> Control.Sequence (a s k0) (k s Unit)
let If cd th el  = fun s k -> k s (Control.If cd (th s k0) (el s k0))
let IfU cd th  = fun s k -> k s (Control.If cd (th s k0) Unit)
//let Match x som non = fun s k -> k s <@ match %x with Some i -> %som <@i@> s k0 | None -> %non s k0 @>
let While c b = fun s k -> k s (Control.While (c s k0) (b s k0))
//let RecLoop g a = fun s k -> k s <@ let rec loop j = %g <@loop@> <@j@> s k0 in loop %a @>
//let RecFun f = fun s k -> k s <@ fun x -> let rec iter i = %f <@iter@> <@i@> s k0 in iter x @>
let Iterate it l f = fun s k -> let n,s' = GetNewVar("i",s) in k s' (it (Function.Lambda n (fun i -> f i s' k0)) l)
let Fun f = fun s k -> let n,s' = GetNewVar("a",s) in k s' (Function.Lambda n (fun a -> f a s' k0))
let Fun2 f = Fun (fun t -> Fun (f t))
let Fun3 f = Fun (fun t -> Fun2 (f t))
let Fun4 f = Fun (fun t -> Fun3 (f t))

// Codegen computation expression
type CGBuilder() =
    inherit SCBuilder()
    member x.Delay f = f()
    member x.Yield a = Return a
    member x.ReturnFrom a = a
    member x.YieldFrom a = a
    member x.Zero() = Return Unit
    member x.For(l,f) = Iterate Seq.Iter l f
    member x.While(c,e) = While (c()) e
    member x.Combine(a,b) = Sequence a b
    member x.Using(m,f) = fun s k -> let n,s' = GetNewVar("t",s) in Control.Let n m (fun v -> f v s' k)
let codegen = CGBuilder()


let DefineOnce n f = codegen {
    let sr = StateRecord("funs_" + n)
    let! c = sr.HasElement()
    if c then yield! sr.Lookup()
    else yield! codegen {
        let! t = fun s k -> Control.Let n f (k s)
        do! sr.Extend t
        return t
    }
}

let DefineOnceRec n f = codegen {
    let sr = StateRecord("funs_" + n)
    let! c = sr.HasElement()
    if c then yield! sr.Lookup()
    else yield! codegen {
        let! t = fun s k -> Control.LetRec n f (k s)
        do! sr.Extend t
        return t
    }
}
