﻿module EA

open System
open Value
open Codegen
open Algebra
open Polynomial
open GBasis
open Bool
open Tuple
open Ref

let SpecialMap2 f z l1 l2 =
    let rec i_map2 = function
    | [],[] -> []
    | [],h::t -> f z h :: i_map2([],t)
    | h::t,[] -> f h z :: i_map2(t,[])
    | h1::t1,h2::t2 -> f h1 h2 :: i_map2(t1,t2)
    let rec trim = function
    | [] -> []
    | h::t when h=z -> trim t
    | l -> l
    i_map2(l1,l2) |> List.rev |> trim |> List.rev

let map2 f z l1 l2 =
    match f,z,l1,l2 with
    | V f,V z,V l1,V l2 -> SpecialMap2 f z l1 l2 |> V
    | _ -> E <@ SpecialMap2 (%GetE f) (%GetE z) (%GetE l1) (%GetE l2) @>

let PolyRing (cr:QuotientRing<'a>) =
    let zero = V []
    let one = List.Singleton cr.one
    let neg = List.Map (UnaryOp.Flatten cr.neg)
    let add = map2 (BinaryOp.Flatten cr.add) cr.zero
    let sub = map2 (BinaryOp.Flatten cr.sub) cr.zero
    let pad i = List.Append (List.Replicate i cr.zero)
    let scale s = List.Map (UnaryOp.Flatten <| cr.mul s)
    let mul x y =
        let parts = List.Mapi (BinaryOp.Flatten <| fun i s -> pad i (scale s y)) x
        List.Fold (BinaryOp.Flatten add) zero parts
    let div p1 p2 =
        let l1,l2 = List.Length p1, List.Length p2
        let c1 = List.Nth p1 (Idx.sub l1 Idx.one)
        let c2 = List.Nth p2 (Idx.sub l2 Idx.one)
        let mkDiv d = Option.Some(pad (Idx.sub l1 l2) (List.Singleton d))
        if cr :? Field<'a> then
            let d = (cr :?> Field<'a>).div c1 c2
            Control.If (l1 ^> l2) (mkDiv d) (V None)
        else
            let d = cr.div c1 c2
            Control.If ((l1 ^> l2) ^&& (Option.IsSome d))
                       (mkDiv(Option.UnOption d)) (V None)
    let rem p1 p2 =
        let l1,l2 = List.Length p1, List.Length p2
        let c1 = List.Nth p1 (Idx.sub l1 Idx.one)
        let c2 = List.Nth p2 (Idx.sub l2 Idx.one)
        let σ =
            if cr :? Field<'a> then
                let d = (cr :?> Field<'a>).div c1 c2
                fun x y -> cr.sub x (cr.mul y d)
            else
                fun x y -> cr.sub (cr.mul x c2) (cr.mul y c1)
        map2 (BinaryOp.Flatten σ) cr.zero p1 (pad (Idx.sub l1 l2) p2)

    QuotientRing.Gen(zero, one, neg, add, mul, Some sub,
        None, None, Some div, Some rem, None, None, None)

let Monomials =
    MonomialMonoid.Gen(
        Idx.zero,
        Idx.add,
        id,
        Seq.Singleton,
        (fun p -> Seq.Singleton (Pair Idx.zero p)),
        Seq.Head,
        (fun p -> Snd (Seq.Head p)),
        (fun x y -> let d = Idx.sub x y
                    Control.If (d ^>= Idx.zero) (Option.Some d) (V None)),
        Min,
        Max,
        (fun x y -> Idx.sub (Max x y) x))

let UnivariatePolynomial cr =
    let tm = TermModule.Generic(cr, Monomials)
    let pr = PolyRing cr
    let mk c m = List.Append (List.Replicate m cr.zero) (List.Singleton c)
    PolynomialAlgebra.Gen(
        cr,
        tm,
        pr,
        List.Singleton,
        Some(fun s -> List.Map (UnaryOp.Flatten (cr.mul s))),
        Compare,
        (fun p -> Seq.Mapi (BinaryOp.Flatten <| fun i n -> tm.make n i) p),
        Some List.Length,
        List.IsEmpty,
        (fun p -> let last = Idx.sub (List.Length p) Idx.one
                  tm.make (List.Nth p last) last),
        Some(fun p -> Idx.sub (List.Length p) Idx.one),
        Some(fun p -> List.Nth p (Idx.sub (List.Length p) Idx.one)),
        List.Reverse >> List.Tail >> List.Reverse,
        (fun ts ->
            let ps = Seq.Map (UnaryOp.Flatten <| fun t -> mk (tm.c t) (tm.m t)) ts
            Seq.Fold (BinaryOp.Flatten pr.add) pr.zero ps),
        mk)


let PairInput<'p> =
    let s1,s2 = StateRecord("p1"),StateRecord("p2")
    {   new Input<'p,'p*'p> with
        member i.Process p = codegen {
            use p1 = Ref (Fst p)
            use p2 = Ref (Snd p)
            do! s1.Extend p1
            do! s2.Extend p2
            return Seq.Empty()
        }
    }

let LastOutput<'p> =
    let s1,s2 = StateRecord("p1"),StateRecord("p2")
    {   new Output<'p,'p> with
        member o.Process _ = codegen {
            let! p2 = s2.Lookup()
            return Deref p2
        }
    }

let PairContainer(A:PolynomialAlgebra<_,_,_,_>) =
    let s1,s2 = StateRecord("p1"),StateRecord("p2")
    {   new Container<_,_,_> with
        member lc.Init s = Prepend <| codegen {
            let! p1 = s1.Lookup()
            let! p2 = s2.Lookup()
            return! IfU (A.deg (Deref p2) ^> A.deg (Deref p1)) (codegen {
                use t = Deref p1
                yield Assign p1 (Deref p2)
                yield Assign p2 t
            })
        }
        member lc.Add _ p = codegen {
            let! p1 = s1.Lookup()
            let! p2 = s2.Lookup()
            return! If (A.deg p ^> A.deg (Deref p2))
                (codegen {
                    yield Assign p1 p
                })
                (codegen {
                    yield Assign p1 (Deref p2)
                    yield Assign p2 p
                })
            return Unit
        }
        member lc.Get _ _ = Return A.zero
        member lc.All _ = Return <| Seq.Empty()
        member lc.Indexes _ = Return <| Seq.Empty()
    }

let PairPick =
    let s1,s2 = StateRecord("p1"),StateRecord("p2")
    {   new WorkingSet<_,_,_> with
        member s.Init _ = Let <| codegen {
            return Ref (E <@ true @>)
        }
        member s.Add w (_,_,_) = codegen {
            return Assign w True
        }
        member s.Del _ (_,_,_) = Return Unit
        member s.HasMore w _ = codegen {
            return Deref w
        }
        member s.Pick w _ = codegen {
            yield Assign w False
            let! p1 = s1.Lookup()
            let! p2 = s2.Lookup()
            return Pair Unit Unit
        }
        member s.All _ _ = Return <| Seq.Empty()
}

let FlagExpansion(S:WorkingSet<_,_,_>) =
    {   new ExpansionStrategy<_,_,_,_> with
        member e.Init(_,_) = Return Unit
        member e.Expand _ (c,s,i) = S.Add s (c,i,i)
    }

let Remainder(A:PolynomialAlgebra<_,_,_,_>) =
    let s1,s2 = StateRecord("p1"),StateRecord("p2")
    {   new SPoly<_,_,_,_> with
        member p.Init _ = Return Unit
        member p.σ _ (_,_,_) = codegen {
            let! p1 = s1.Lookup()
            let! p2 = s2.Lookup()
            return A.rem (Deref p1) (Deref p2)
        }
    }


let ea =
    let t = Debug.NoTrace
    let bk = StandardBasis
    let K = Field.QQ
    let pa = UnivariatePolynomial K
    let ip = PairInput
    let pc = PairContainer pa
    let ws = PairPick
    let es = FlagExpansion ws
    let sp = Remainder pa
    let nr = NormalRemainder.NoRemainder
    let rs = ReductionStrategy.NoReduction
    let cf = CanonicalForm.NoOperation
    let op = LastOutput
    GBSolver(t,bk,pa,ip,pc,ws,es,sp,nr,rs,cf,op)

(*
#r "bin/Debug/Groebner.dll";;
#r "FSharp.PowerPack.dll";;
#r "FSharp.PowerPack.Linq.dll";;
fsi.AddPrinter Quote.Print;;
open Value;;
open Bool;;
open Tuple;;
open Ref;;
open Codegen;;
open Algebra;;
open Polynomial;;
open GBasis;;
open EA;;

let t = Debug.ConsoleTrace 5
let bk = StandardBasis
let K = Field.QQ
let pa = UnivariatePolynomial K
let ip = PairInput
let pc = PairContainer pa
let ws = PairPick
let es = FlagExpansion ws
let sp = Remainder pa
let nr = NormalRemainder.NoRemainder
let rs = ReductionStrategy.NoReduction
let cf = CanonicalForm.NoOperation
let op = LastOutput
let qea = GBSolver(Debug.NoTrace,bk,pa,ip,pc,ws,es,sp,nr,rs,cf,op)
let ea = GetV(GBSolver(t,bk,pa,ip,pc,ws,es,sp,nr,rs,cf,op));;

let p1 = [6N;7N;1N];;
let p2 = [-6N;-5N;1N];;
let p3 = [-5N;2N;8N;-3N;-3N;0N;1N;0N;1N];;
let p4 = [21N;-9N;-4N;0N;5N;0N;3N];;
*)
