﻿module GE

open Microsoft.FSharp.Collections
open System
open Value
open Codegen
open Algebra
open Polynomial
open GBasis
open Bool
open Tuple

let mkConst n z v = List.Append (List.Replicate (V n) z) (List.Singleton v)

let PolyRing n (c:#Ring<_>) =
    QuotientRing.Gen(
        mkConst n c.zero c.zero,
        mkConst n c.zero c.one,
        List.Map  (UnaryOp.Flatten c.neg),
        List.Map2 (BinaryOp.Flatten c.add),
        (fun x y -> failwith "Undefined multiplication operation"),
        List.Map2 (BinaryOp.Flatten c.sub) |> Some,
        None, None, None, None, None, None, None)

let Monomials n =
    MonomialMonoid.Gen(
        V n,
        Min,
        (fun l -> Control.If (l ^= (V n)) Idx.zero Idx.one),
        (fun l -> Seq.Map (UnaryOp.Flatten <| fun i -> Control.If (l ^= i) Idx.one Idx.zero) (Seq.Make Idx.zero (V(n-1)))),
        (fun l -> Control.If (l ^= (V n)) (Seq.Empty()) (Seq.Singleton <| Pair l Idx.one)),
        (fun l -> Seq.FindIndex (UnaryOp.Flatten <| Bool.lt Idx.zero) (Seq.Append l <| Seq.Singleton Idx.one)),
        (fun l -> Control.If (Seq.IsEmpty l) (V n) (Fst <| Seq.Head l)),
        (fun x y -> Control.If (x ^= y) (Option.Some (V n)) (V None)),
        (fun x y -> Control.If (x ^= y) x (V n)),
        (fun x y -> Control.If (x ^= y) x (Idx.neg Idx.one)),
        (fun x y -> Control.If (x ^= y) (V n) (Idx.neg Idx.one)))

let LinearPolynomial n c =
    let tm = TermModule.Generic(c, Monomials n)
    let pr = PolyRing n c
    let notzero = UnaryOp.Flatten (Bool.neq c.zero)
    PolynomialAlgebra.Gen(
        c,
        tm,
        pr,
        mkConst n c.zero,
        Some(fun s -> List.Map (UnaryOp.Flatten (c.mul s))),
        Compare,
        (fun p -> Seq.Mapi (BinaryOp.Flatten <| fun i n -> tm.make n i) p),
        None,
        List.Forall (UnaryOp.Flatten <| Bool.eq c.zero),
        (fun p -> Generate <| codegen {
            use i = List.FindIndex notzero p
            return tm.make (List.Nth p i) i
        }),
        List.FindIndex notzero |> Some,
        List.Find notzero |> Some,
        (fun p -> List.SetNth p (List.FindIndex notzero p) c.zero),
        Seq.Fold (BinaryOp.Flatten <| fun p t -> List.SetNth p (tm.m t) (tm.c t)) pr.zero,
        fun c m -> List.SetNth pr.zero m c)

let LM (A:PolynomialAlgebra<_,_,_,_>) p = codegen {
    let! lm = DefineOnce "lm" <| UnaryOp.Flatten A.LM
    return Function.Apply lm p
}

let Compatible (A:PolynomialAlgebra<_,_,_,_>,C:Container<_,_,_>) (c,i,j) = codegen {
    let! c_i = C.Get c i
    let! c_j = C.Get c j
    let! lm_i = LM A c_i
    let! lm_j = LM A c_j
    return (i ^<> j) ^&& (lm_i ^= lm_j)
}

let GetResidual(A:PolynomialAlgebra<_,_,_,_>) =
    let res p q i =
        let lc1 = List.Nth p i
        let lc2 = List.Nth q i
        let t1 = A.scalar (A.CR.τ lc1 lc2) p
        let t2 = A.scalar (A.CR.τ lc2 lc1) q
        A.sub t1 t2
    DefineOnce "residual" <| TernaryOp.Flatten res

let AddPairs(A:PolynomialAlgebra<_,_,_,_>,C:Container<_,_,_>,S:WorkingSet<_,_,_>) =
    {   new ExpansionStrategy<_,_,_,_> with
        member e.Init(c,s) = Prepend <| codegen {
            let! _ = LM A A.zero
            let! ids = C.Indexes c
            for p in Seq.AllPairs ids do
                let i,j = Fst p, Snd p
                let! comp = Compatible (A,C) (c,i,j)
                yield! IfU comp <| S.Add s (c,i,j)
        }
        member e.Expand _ (c,s,k) = codegen {
            let! ids = C.Indexes c
            for i in ids do
                let! comp = Compatible (A,C) (c,i,k)
                yield! IfU comp <| S.Add s (c,i,k)
        }
    }

let Residual(A:PolynomialAlgebra<_,_,_,_>,C:Container<_,_,_>) =
    {   new SPoly<_,_,_,_> with
        member p.Init _ = GetResidual A
        member p.σ res (c,i,j) = codegen {
            let! c_i = C.Get c i
            let! c_j = C.Get c j
            let! lm_i = LM A c_i
            return Function.Apply3 res c_i c_j lm_i
        }
    }

let PruneRows(A:PolynomialAlgebra<_,_,_,_>,C:Container<_,_,_>) =
    {   new ReductionStrategy<_,_,_> with
        member s.Init _ = Let <| Return (List.Empty())
        member s.Reduce rs (c,i) = codegen {
            let! red = Fun <| fun j -> codegen {
                let! comp = Compatible (A,C) (c,i,j)
                let seen = List.Contains rs j
                return (Not seen) ^&& comp
            }
            let! ids = C.Indexes c
            yield! If (Seq.Exists red ids)
                      (codegen {yield List.Add rs i; return True})
                      (Return False)
        }
    }

let BackPropagate(A:PolynomialAlgebra<'a,_,_,_>,C:Container<_,_,_>) =
    {   new CanonicalForm<_,_,_> with
        member p.Init _ = GetResidual A
        member f.Canonicalize r (c,p) = codegen {
            let! qs = C.All c

            // Back-propagate
            let! rem = Fun2 <| fun p q -> codegen {
                let! lm_p = LM A p
                let! lm_q = LM A q
                yield Control.If (lm_q ^> lm_p) (Function.Apply3 r p q lm_q) p
            }
            use cf = Seq.Fold rem p qs

            // Make the polynomials monic
            let! div = Fun <| fun c ->
                if A.CR :? Field<'a>
                then codegen {
                    let f = A.CR :?> Field<'a>
                    use q = A.LC cf
                    return f.div c q
                    }
                else codegen {
                    use q = List.Fold (BinaryOp.Flatten A.CR.gcd) (A.LC cf) cf
                    return Option.UnOption(A.CR.div c q)
                }
            return List.Map div cf
        }
    }

let ge =
    let t = Debug.NoTrace
    let bk = ReducedBasis
    let K = Field.QQ
    let pa = LinearPolynomial 4 K
    let ip = Input.InBasis
    let pc = Container.ListContainer
    let ws = WorkingSet.DirectPick
    let es = ExpansionStrategy.DirectExpand(pc,ws)
    let sp = Residual(pa,pc)
    let nr = NormalRemainder.NoRemainder
    let rs = ReductionStrategy.EliminateDivisors(pa,pc)
    let cf = BackPropagate(pa,pc)
    let op = Output.OutBasis
    let qge = GBSolver(t,bk,pa,ip,pc,ws,es,sp,nr,rs,cf,op)
    GetV qge
//*)
(*
#r "bin/Debug/Groebner.dll";;
#r "FSharp.PowerPack.dll";;
#r "FSharp.PowerPack.Linq.dll";;
fsi.AddPrinter Quote.Print;;
open Value;;
open Codegen;;
open Algebra;;
open Polynomial;;
open GBasis;;
open GE;;

let f1 = [2;1;-1;8]
let f2 = [3;1;-2;11]
let f3 = [2;-1;-2;3];;

let p1 = [ 2N;-1N;-1N;-1N;-1N]
let p2 = [ 1N;-1N;-1N; 0N; 0N]
let p3 = [ 0N; 3N;-2N; 1N;-4N]
let p4 = [ 1N; 3N; 0N;-2N;-3N]
;;

let t = Debug.ConsoleTrace 5
let bk = ReducedBasis
let K = Field.QQ
let pa = LinearPolynomial 4 K
let ip = Input.InBasis
let pc = Container.ListContainer
let ws = WorkingSet.DirectPick
let es = AddPairs(pa,pc,ws)
let sp = Residual(pa,pc)
let nr = NormalRemainder.NoRemainder
let rs = PruneRows(pa,pc)
let cf = BackPropagate(pa,pc)
let op = Output.OutBasis
let qge = GBSolver(Debug.NoTrace,bk,pa,ip,pc,ws,es,sp,nr,rs,cf,op)
let ge = GetV(GBSolver(t,bk,pa,ip,pc,ws,es,sp,nr,rs,cf,op));;

ge[p1;p2;p3;p4];;

let t = Debug.NoTrace//ConsoleTrace 5
let bk = ReducedBasis
let K = Field.QQ
let pa = LinearPolynomial 3 K
let ip = Input.InBasis
let pc = Container.ListContainer
let ws = WorkingSet.DirectPick
let es = ExpansionStrategy.DirectExpand(pc,ws)
let sp = Residual(pa,pc)
let nr = Remainder(pa,pc)
let rs = ReductionStrategy.EliminateDivisors(pa,pc)
let cf = BackPropagate(pa,pc)
let op = Output.OutBasis
let qge = GBSolver(t,bk,pa,ip,pc,ws,es,sp,nr,rs,cf,op)
let ge2 = GetV qge;;

let f4 = [2N;1N;-1N;8N]
let f5 = [-3N;-1N;2N;-11N]
let f6 = [-2N;1N;2N;-3N];;
ge2[f4;f5;f6];;

let f1 = [2;0;0;8]
let f2 = [0;6;0;8]
let f3 = [0;0;12;-3];;
ge[f1;f2;f3];;

let f1 = [2N;0N;0N;8N]
let f2 = [0N;6N;0N;8N]
let f3 = [0N;0N;12N;-3N];;
ge2[f1;f2;f3];;


 [  2  1 -1 |  8 ]      [ 1 1/3 -2/3 | 11/3 ]      [ 1 0 0 |  2 ]
 [ -3 -1  2 | 11 ]  ->  [ 0  1   2/5 | 13/5 ]  ->  [ 0 1 0 |  3 ]
 [ -2  1  2 | -3 ]      [ 0  0   1   | -1   ]      [ 0 0 1 | -1 ]

*)
